<?php

/**
 * Streamit\Utility\Styles\Component class
 *
 * @package streamit
 */

namespace Streamit\Utility\Styles;

use function Streamit\Utility\streamit;
use Streamit\Utility\Component_Interface;
use Streamit\Utility\Templating_Component_Interface;

/**
 * Class for managing stylesheets.
 *
 * Exposes template tags:
 * * `streamit()->print_styles()`
 */

class Component implements Component_Interface, Templating_Component_Interface
{

	/**
	 * Associative array of CSS files, as $handle => $data pairs.
	 * $data must be an array with keys 'file' (file path relative to 'assets/css' directory), and optionally 'global'
	 * (whether the file should immediately be enqueued instead of just being registered) and 'preload_callback'
	 * (callback function determining whether the file should be preloaded for the current request).
	 *
	 * Do not access this property directly, instead use the `get_css_files()` method.
	 *
	 * @var array
	 */
	protected $css_files;

	protected $dequeue_css_files =  array();

	protected $base_url;
	protected $shop_slugs;

	/**
	 * Associative array of Google Fonts to load, as $font_name => $font_variants pairs.
	 *
	 * Do not access this property directly, instead use the `get_google_fonts()` method.
	 *
	 * @var array
	 */
	protected $google_fonts;

	/**
	 * Gets the unique identifier for the theme component.
	 *
	 * @return string Component slug.
	 */
	public function get_slug(): string
	{
		return 'styles';
	}

	public function __construct()
	{
		$page_url 			= $_SERVER['REQUEST_URI'];
		$this->base_url 	= basename($page_url);
		$url_parts 			= explode('/', $page_url);
		$url_parts 			= array_filter($url_parts);

		foreach ($url_parts as $key => $url_part) {
			// if ($key > 1) { // CONDITION FOR LIVE PATH
				if ($url_part == "movie-playlist" || $url_part == "tv-show-playlist" ||	$url_part == "video-playlist" || $url_part == "product" || $url_part = "product-category" || $url_part = "product-tag" || $url_part == "checkout" || $url_part == "movie" || $url_part == "tv-show" || $url_part == "video")
					$type = $url_part;
			// }
		}

		$end_url = (isset($type) && !empty($type)) ? end($url_parts) : '';

		$this->shop_slugs = [
			'shop',
			'shop-2',
			'merchandise-store',
			'store',
			'product',
			'cart-page',
			'cart',
			'checkout-page',
			'checkout',
			'my-account-page',
			'my-account',
			$end_url
		];
	}

	/**
	 * Adds the action and filter hooks to integrate with WordPress.
	 */
	public function initialize()
	{
		add_action('wp_enqueue_scripts', array($this, 'action_enqueue_styles'));
		add_action('wp_head', array($this, 'action_preload_styles'));
		add_action('after_setup_theme', array($this, 'action_add_editor_styles'));
		add_filter('wp_resource_hints', array($this, 'filter_resource_hints'), 10, 2);
		add_action("admin_enqueue_scripts", function () {
			wp_enqueue_style("streamit-font-awesome", get_template_directory_uri() . "/assets/css/vendor/font-awesome/css/all.min.css", [], '1.0.1', "all");
		});

		add_action('wp_enqueue_scripts', array($this, 'action_dequeue_styles'), 999);
	}

	/**
	 * Gets template tags to expose as methods on the Template_Tags class instance, accessible through `streamit()`.
	 *
	 * @return array Associative array of $method_name => $callback_info pairs. Each $callback_info must either be
	 *               a callable or an array with key 'callable'. This approach is used to reserve the possibility of
	 *               adding support for further arguments in the future.
	 */

	public function template_tags(): array
	{
		return array(
			'print_styles' => array($this, 'print_styles'),
		);
	}

	/**
	 * Registers or enqueues stylesheets.
	 *
	 * Stylesheets that are global are enqueued. All other stylesheets are only registered, to be enqueued later.
	 */

	public function action_enqueue_styles()
	{
		// Enqueue Google Fonts.
		$google_fonts_url = $this->get_google_fonts_url();
		if (!empty($google_fonts_url)) {
			wp_enqueue_style('streamit-fonts', $google_fonts_url, array(), null); // phpcs:ignore WordPress.WP.EnqueuedResourceParameters.MissingVersion
		}

		$css_uri 					= get_template_directory_uri() . '/assets/css/';
		$css_dir 					= get_template_directory() . '/assets/css/';
		$preloading_styles_enabled 	= $this->preloading_styles_enabled();

		$css_files = $this->get_css_files();
		foreach ($css_files as $handle => $data) {
			$src     	 = $css_uri . $data['file'];
			$is_external = isset($data['is_external']) ? $data['is_external'] : true;

			/*
			 * Enqueue global stylesheets immediately and register the other ones for later use
			 * (unless preloading stylesheets is disabled, in which case stylesheets should be immediately
			 * enqueued based on whether they are necessary for the page content).
			 */

			if ($data['global'] || !$preloading_styles_enabled && is_callable($data['preload_callback']) && call_user_func($data['preload_callback'])) {
				wp_enqueue_style($handle, $src, array(), STREAMIT_THEME_VERSION, $data['media']);
			} else {
				wp_register_style($handle, $src, array(), STREAMIT_THEME_VERSION, $data['media']);
			}

			wp_style_add_data($handle, 'precache', true);
		}
	}

	public function action_dequeue_styles()
	{
		if (class_exists("MasVideos") && !in_array($this->base_url, $this->shop_slugs)) {
			$this->dequeue_css_files = [
				'masvideos-general',
				'masvideos-inline',
			];
		}

		if (class_exists('WooCommerce') && !in_array($this->base_url, $this->shop_slugs)) {
			$husky_css = [
				"woof",
				"icheck-jquery-color-flat",
				"icheck-jquery-color-square",
				"icheck-jquery-color-minimal",
				"woof_by_author_html_items",
				"woof_by_instock_html_items",
				"woof_by_onsales_html_items",
				"woof_label_html_items",
				"woof_select_radio_check_html_items",
				"woof_sd_html_items_checkbox",
				"woof_sd_html_items_radio",
				"woof_sd_html_items_switcher",
				"woof_sd_html_items_color",
				"woof_sd_html_items_tooltip",
				"woof_sd_html_items_front",
				"woof-switcher23",
				"chosen-drop-down",
				"woof_by_text_html_items",
				"woof_turbo_mode_html_items",

				"woof_sections_style",
				"woof_tooltip-css",
				"woof_tooltip-css-noir",
				"ion.range-slider",
				"woof-front-builder-css",
				"woof-front-builder-popup-23",
				"woof-front-builder-table",
				"woof-slideout-tab-css",
				"woof-slideout-css"
			];

			$wpc_css = [
				"perfect-scrollbar",
				"perfect-scrollbar-wpc",
				"magnific-popup",
				"woosq-feather",
				"woosq-frontend",
			];

			$yith_css = [
				"jquery-selectBox",
				"yith-wcwl-font-awesome",
				"yith-wcwl-main"
			];

			$woocommmerce_css = [
				'wc-blocks-vendors-style',
				'wc-all-blocks-style',
				'wc-blocks-style',
				'woocommerce_prettyPhoto_css',
				'woocommerce-layout',
				'woocommerce-smallscreen',
				'woocommerce-general',
				'woocommerce-inline',
			];

			$this->dequeue_css_files = array_merge($husky_css, $wpc_css, $yith_css, $woocommmerce_css, $this->dequeue_css_files);
		}

		foreach ($this->dequeue_css_files as $handle) {
			wp_dequeue_style($handle);
		}
	}

	/**
	 * Preloads in-body stylesheets depending on what templates are being used.
	 *
	 * Only stylesheets that have a 'preload_callback' provided will be considered. If that callback evaluates to true
	 * for the current request, the stylesheet will be preloaded.
	 *
	 * Preloading is disabled when AMP is active, as AMP injects the stylesheets inline.
	 *
	 * @link https://developer.mozilla.org/en-US/docs/Web/HTML/Preloading_content
	 */

	public function action_preload_styles()
	{

		// If preloading styles is disabled, return early.
		if (!$this->preloading_styles_enabled()) {
			return;
		}

		$wp_styles = wp_styles();

		$css_files = $this->get_css_files();
		foreach ($css_files as $handle => $data) {

			// Skip if stylesheet not registered.
			if (!isset($wp_styles->registered[$handle])) {
				continue;
			}

			// Skip if no preload callback provided.
			if (!is_callable($data['preload_callback'])) {
				continue;
			}

			// Skip if preloading is not necessary for this request.
			if (!call_user_func($data['preload_callback'])) {
				continue;
			}

			$preload_uri = $wp_styles->registered[$handle]->src . '?ver=' . $wp_styles->registered[$handle]->ver;

			echo '<link rel="preload" id="' . esc_attr($handle) . '-preload" href="' . esc_url($preload_uri) . '" as="style">';
			echo "\n";
		}
	}

	/**
	 * Enqueues WordPress theme styles for the editor.
	 */
	public function action_add_editor_styles()
	{
		// Enqueue Google Fonts.
		$google_fonts_url = $this->get_google_fonts_url();
		if (!empty($google_fonts_url)) {
			add_editor_style($this->get_google_fonts_url());
		}

		// Enqueue block editor stylesheet.
		add_editor_style('assets/css/editor/editor-styles.min.css');
	}

	/**
	 * Adds preconnect resource hint for Google Fonts.
	 *
	 * @param array  $urls          URLs to print for resource hints.
	 * @param string $relation_type The relation type the URLs are printed.
	 * @return array URLs to print for resource hints.
	 */
	public function filter_resource_hints(array $urls, string $relation_type): array
	{
		if ('preconnect' === $relation_type && wp_style_is('streamit-fonts', 'queue')) {
			$urls[] = array(
				'href' => 'https://fonts.gstatic.com',
				'crossorigin',
			);
		}

		return $urls;
	}

	/**
	 * Prints stylesheet link tags directly.
	 *
	 * This should be used for stylesheets that aren't global and thus should only be loaded if the HTML markup
	 * they are responsible for is actually present. Template parts should use this method when the related markup
	 * requires a specific stylesheet to be loaded. If preloading stylesheets is disabled, this method will not do
	 * anything.
	 *
	 * If the `<link>` tag for a given stylesheet has already been printed, it will be skipped.
	 *
	 * @param string ...$handles One or more stylesheet handles.
	 */

	public function print_styles(string ...$handles)
	{
		// If preloading styles is disabled (and thus they have already been enqueued), return early.
		if (!$this->preloading_styles_enabled()) {
			return;
		}

		$css_files = $this->get_css_files();
		$handles   = array_filter(
			$handles,
			function ($handle) use ($css_files) {
				$is_valid = isset($css_files[$handle]) && !$css_files[$handle]['global'];
				if (!$is_valid) {
					/* translators: %s: stylesheet handle */
					_doing_it_wrong(__CLASS__ . '::print_styles()', esc_html(sprintf(__('Invalid theme stylesheet handle: %s', 'streamit'), $handle)), 'Streamit 2.0.0');
				}
				return $is_valid;
			}
		);

		if (empty($handles)) {
			return;
		}

		wp_print_styles($handles);
	}

	/**
	 * Determines whether to preload stylesheets and inject their link tags directly within the page content.
	 *
	 * Using this technique generally improves performance, however may not be preferred under certain circumstances.
	 * For example, since AMP will include all style rules directly in the head, it must not be used in that context.
	 * By default, this method returns true unless the page is being served in AMP. The
	 * {@see 'streamit_preloading_styles_enabled'} filter can be used to tweak the return value.
	 *
	 * @return bool True if preloading stylesheets and injecting them is enabled, false otherwise.
	 */

	protected function preloading_styles_enabled()
	{
		$preloading_styles_enabled = !streamit()->is_amp();

		/**
		 * Filters whether to preload stylesheets and inject their link tags within the page content.
		 *
		 * @param bool $preloading_styles_enabled Whether preloading stylesheets and injecting them is enabled.
		 */
		return apply_filters('streamit_preloading_styles_enabled', $preloading_styles_enabled);
	}

	/**
	 * Gets all CSS files.
	 *
	 * @return array Associative array of $handle => $data pairs.
	 */

	protected function get_css_files(): array
	{
		global $streamit_options;

		if (is_array($this->css_files)) {
			return $this->css_files;
		}

		$presetopt = streamit()->streamit_is_rtl() ?  'bootstrap.rtl'  : 'bootstrap';

		/* end layout style swticher option */
		$css_files = array(
			'slick' => array(
				'file'   		=> 'vendor/slick/slick.css',
				'global' 		=> true,
				'is_external'	=> true,
			),
			'select2'			=> array(
				'file'   		=> 'vendor/select2.min.css',
				'global' 		=> true,
				'is_external'	=> true,
			),
			'magnific-popup'	=> array(
				'file'   		=> 'vendor/magnific-popup.css',
				'global' 		=> true,
				'is_external'	=> true,
			),
			'bootstrap' => array(
				'file'   		=> 'vendor/' . $presetopt . '.min.css',
				'global' 		=> true,
				'is_external'	=> true,
			),
			'streamit-font-awesome' => array(
				'file'   		=> 'vendor/font-awesome/css/all.min.css',
				'global' 		=> true,
				'is_external'	=> true,
			),
			'animate-min' => array(
				'file'   		=> 'vendor/animate.min.css',
				'global' 		=> true,
				'is_external'	=> true,
			),
			'streamit-style'    	=> array(
				'file'          => 'streamit-style.min.css',
				'global' 		=> true,
				'is_external'	=> false,
			),
			'streamit-responsive' => array(
				'file'          => 'streamit-responsive.min.css',
				'global' 		=> true,
				'is_external'	=> false,
			),
			'streamit-menu-burger' => array(
				'file'          => 'burger-menu.min.css',
				'global' 		=> true,
				'is_external'	=> false,
			),
			'streamit-select2' => array(
				'file'   		=> 'select2.min.css',
				'global' 		=> true,
				'is_external'	=> true,
			),
		);

		if (class_exists('WooCommerce') && in_array($this->base_url, $this->shop_slugs)) {
			$woo_css = array(
				'streamit-woocommerce' => array(
					'file'   		=> 'woocommerce.min.css',
					'global' 		=> true,
					'is_external'	=> false,
				)
			);

			$css_files = array_merge($css_files, $woo_css);
		}

		$is_rtl = isset($_COOKIE['theme_scheme_direction']) && $_COOKIE['theme_scheme_direction'] == "rtl" ? true : isset($streamit_options['streamit_layout_mode_options']) && $streamit_options['streamit_layout_mode_options'] == "2";
		if ($is_rtl) {
			$rtl_css = array(
				'streamit-rtl'    	=> array(
					'file'          => 'rtl.min.css',
					'global' 		=> true,
					'is_external'	=> false,
				),
			);

			$css_files = array_merge($css_files, $rtl_css);
		}

		if (class_exists("MasVideos") && in_array($this->base_url, $this->shop_slugs)) {
			$masvideos_css = array(
				'streamit-swiper-bundle' => array(
					'file'   		=> 'vendor/swiper-bundle.min.css',
					'global' 		=> true,
					'is_external'	=> true,
				),
				'streamit-swiper-general' => array(
					'file'   		=> 'swiper-general.min.css',
					'global' 		=> true,
					'is_external'	=> true,
				),
			);

			$css_files = array_merge($css_files, $masvideos_css);
		}

		/**
		 * Filters default CSS files.
		 *
		 * @param array $css_files Associative array of CSS files, as $handle => $data pairs.
		 *                         $data must be an array with keys 'file' (file path relative to 'assets/css'
		 *                         directory), and optionally 'global' (whether the file should immediately be
		 *                         enqueued instead of just being registered) and 'preload_callback' (callback)
		 *                         function determining whether the file should be preloaded for the current request).
		 */

		$css_files = apply_filters('streamit_css_files', $css_files);

		$this->css_files = array();
		foreach ($css_files as $handle => $data) {
			if (is_string($data)) {
				$data = array('file' => $data);
			}

			if (empty($data['file'])) {
				continue;
			}

			$this->css_files[$handle] = array_merge(
				array(
					'global'           => false,
					'preload_callback' => null,
					'media'            => 'all',
				),
				$data
			);
		}

		return $this->css_files;
	}

	/**
	 * Returns Google Fonts used in theme.
	 *
	 * @return array Associative array of $font_name => $font_variants pairs.
	 */

	protected function get_google_fonts(): array
	{
		if (is_array($this->google_fonts)) {
			return $this->google_fonts;
		}
		$google_fonts = array(
			'Roboto'     => array('400', '400i', '500', '500i', '600', '700'),
		);

		/**
		 * Filters default Google Fonts.
		 *
		 * @param array $google_fonts Associative array of $font_name => $font_variants pairs.
		 */
		$this->google_fonts = (array) apply_filters('streamit_google_fonts', $google_fonts);

		return $this->google_fonts;
	}

	/**
	 * Returns the Google Fonts URL to use for enqueuing Google Fonts CSS.
	 *
	 * Uses `latin` subset by default. To use other subsets, add a `subset` key to $query_args and the desired value.
	 *
	 * @return string Google Fonts URL, or empty string if no Google Fonts should be used.
	 */

	protected function get_google_fonts_url(): string
	{
		$google_fonts = $this->get_google_fonts();

		if (empty($google_fonts)) {
			return '';
		}

		$font_families = array();

		foreach ($google_fonts as $font_name => $font_variants) {
			if (!empty($font_variants)) {
				if (!is_array($font_variants)) {
					$font_variants = explode(',', str_replace(' ', '', $font_variants));
				}

				$font_families[] = $font_name . ':' . implode(',', $font_variants);
				continue;
			}

			$font_families[] = $font_name;
		}

		$query_args = array(
			'family'  => implode('%7C', $font_families),
			'display' => 'swap',
		);

		return add_query_arg($query_args, 'https://fonts.googleapis.com/css');
	}
}
