<?php

/**
 * Streamit\Utility\Scripts\Component class
 *
 * @package streamit
 */

namespace Streamit\Utility\Scripts;

use function Streamit\Utility\streamit;
use Streamit\Utility\Component_Interface;

class Component implements Component_Interface
{

	/**
	 * Associative array of CSS files, as $handle => $data pairs.
	 * $data must be an array with keys 'file' (file path relative to 'assets/css' directory), and optionally 'global'
	 * (whether the file should immediately be enqueued instead of just being registered) and 'preload_callback'
	 * (callback function determining whether the file should be preloaded for the current request).
	 *
	 * Do not access this property directly, instead use the `get_css_files()` method.
	 *
	 * @var array
	 */
	protected $js_files;

	protected $dequeue_js_files = array();

	protected $base_url;
	protected $shop_slugs;

	/**
	 * Gets the unique identifier for the theme component.
	 *
	 * @return string Component slug.
	 */
	public function get_slug(): string
	{
		return 'scripts';
	}

	public function __construct()
	{
		$page_url 			= $_SERVER['REQUEST_URI'];
		$this->base_url 	= basename($page_url);
		$url_parts 			= explode('/', $page_url);
		$url_parts 			= array_filter($url_parts);

		foreach ($url_parts as $key => $url_part) {
			if ($key > 1) { // CONDITION FOR LIVE PATH
				if ($url_part == "movie-playlist" || $url_part == "tv-show-playlist" ||	$url_part == "video-playlist" || $url_part == "product" || $url_part == "checkout" || $url_part == "movie" || $url_part == "tv-show" || $url_part == "video")
					$type = $url_part;
			}
		}

		$end_url = (isset($type) && !empty($type)) ? end($url_parts) : '';

		$this->shop_slugs = [
			'shop',
			'shop-2',
			'merchandise-store',
			'store',
			'product',
			'cart-page',
			'cart',
			'checkout-page',
			'checkout',
			'my-account-page',
			'my-account',
			$end_url
		];
	}

	/**
	 * Adds the action and filter hooks to integrate with WordPress.
	 */
	public function initialize()
	{
		add_action('wp_enqueue_scripts', array($this, 'action_enqueue_scripts'));
		add_action('wp_footer', 		 array($this, 'action_dequeue_scripts'), 999);
	}

	/**
	 * Registers or enqueues stylesheets.
	 *
	 * Stylesheets that are global are enqueued. All other stylesheets are only registered, to be enqueued later.
	 */
	public function action_enqueue_scripts()
	{
		global $wp_query;
		$js_uri = get_template_directory_uri() . '/assets/js/';
		$js_dir = get_template_directory() . '/assets/js/';
		$js_files = $this->get_js_files();
		wp_enqueue_script('underscore');

		foreach ($js_files as $handle => $data) {
			$src     = $js_uri . $data['file'];
			$version = streamit()->get_asset_version($js_dir . $data['file']);
			wp_enqueue_script($handle, $src, $data['dependency'], $version, $data['mode']);
		}

		wp_localize_script('streamit-loadmore', 'streamit_loadmore_params', array(
			'ajaxurl' 			=> admin_url('admin-ajax.php'),
			'posts' 			=> json_encode($wp_query->query_vars),
			'query'				=> json_encode($wp_query->query),
			'current_page' 		=> get_query_var('paged') ? get_query_var('paged') : 1,
			'max_page' 			=> $wp_query->max_num_pages,
			'template_dir' 		=> get_template_directory_uri(),
			'version'			=> STREAMIT_THEME_VERSION,
			'query1'			=> json_encode($wp_query->query),
		));
		wp_enqueue_script('streamit-loadmore');
	}

	/**
	 * Gets all JS files.
	 *
	 * @return array Associative array of $handle => $data pairs.
	 */
	protected function get_js_files(): array
	{
		if (is_array($this->js_files)) {
			return $this->js_files;
		}

		$js_files = array(
			'slick-min'     	=> array(
				'file'   		=> 'vendor/slick/slick.min.js',
				'dependency' 	=> array('jquery'),
				'mode'			=> [
					'strategy'  => 'async',
					'in_footer' => true,
				],
			),
			'slick-animation'   => array(
				'file'   		=> 'vendor/slick/slick-animation.min.js',
				'dependency' 	=> array('jquery'),
				'mode'			=> [
					'strategy'  => 'async',
					'in_footer' => true,
				],
			),
			'select2'     		=> array(
				'file'   		=> 'vendor/select2.min.js',
				'dependency' 	=> array('jquery'),
				'mode'			=> [
					'strategy'  => 'defer',
					'in_footer' => true,
				],
			),
			'magnific-popup'    => array(
				'file'   		=> 'vendor/jquery.magnific-popup.min.js',
				'dependency' 	=> array('jquery'),
				'mode'			=> [
					'strategy'  => 'async',
					'in_footer' => true,
				],
			),
			'bootstrap' 		=> array(
				'file'   		=> 'vendor/bootstrap.min.js',
				'dependency' 	=> array('jquery'),
				'mode'			=> [
					'strategy'  => 'async',
					'in_footer' => true,
				],
			),
			'streamit-custom'  => array(
				'file'   		=> 'custom.min.js',
				'dependency' 	=> array('jquery'),
				'mode'			=> [
					'strategy'  	=> 'defer',
					'in_footer' => true,
				]
			),
			'streamit-loadmore' 	=> array(
				'file'   		 	=> 'loadmore.min.js',
				// 'file'   		 	=> 'src/loadmore.js',
				'dependency' 	 	=> array('jquery'),
				'mode'				=> [
					'strategy'  	=> 'defer',
					'in_footer' 	=> true,
				],
			),
			'select2-streamit'    => array(
				'file'   		=> 'vendor/select2.min.js',
				'dependency' 	=> array('jquery'),
				'mode'			=> [
					'strategy'  => 'defer',
					'in_footer' => true,
				],
			),
		);

		if (class_exists("MasVideos") && in_array($this->base_url, $this->shop_slugs)) {
			$masvideos_js = array(
				'streamit-swiper-bundle' 	=> array(
					'file'   				=> 'vendor/swiper-bundle.min.js',
					'dependency' 			=> array('jquery'),
					'mode'					=> [
						'strategy'  	=> 'async',
						'in_footer' 	=> true,
					],
				),
				'streamit-swiper-general' 	=> array(
					'file'   				=> 'swiper-general.min.js',
					'dependency' 			=> array('jquery'),
					'mode'					=> [
						'strategy'  	=> 'async',
						'in_footer' 	=> true,
					],
				),
			);

			$js_files = array_merge($js_files, $masvideos_js);
		}

		$this->js_files = array();
		foreach ($js_files as $handle => $data) {
			if (is_string($data)) {
				$data = array('file' => $data);
			}

			if (empty($data['file'])) {
				continue;
			}

			$this->js_files[$handle] = array_merge(
				array(
					'global'           => false,
					'preload_callback' => null,
					'media'            => 'all',
				),
				$data
			);
		}

		return $this->js_files;
	}

	public function action_dequeue_scripts()
	{
		$woof_scripts = $woocommmerce_scripts = $wpc_scripts = $yith_js = array();

		$remove_woof_scripts 			= apply_filters('streamit_remove_woof_scripts', 		 '__true');
		$remove_yith_scripts 			= apply_filters('streamit_remove_yith_scripts', 		 '__true');
		$remove_quickview_scripts 		= apply_filters('streamit_remove_quickview_scripts', 	 '__true');
		$remove_woocommerce_scripts 	= apply_filters('streamit_remove_woocommerce_scripts', '__true');

		if (class_exists("MasVideos") && !in_array($this->base_url, $this->shop_slugs)) {
			$this->dequeue_js_files = [
				'masvideos-gallery-flip',
			];
		}

		if (class_exists('WooCommerce') && !in_array($this->base_url, $this->shop_slugs)) {

			if ($remove_woof_scripts === "__true") {
				$woof_scripts = [
					// 'woof-husky',
					'woof_by_author_html_items',
					'woof_by_instock_html_items',
					'woof_by_onsales_html_items',
					'woof_label_html_items',
					'woof_sections_html_items',
					'woof_select_radio_check_html_items',
					'woof_sd_html_items',
					'woof-slideout-js',
					'woof-slideout-init',

					//footer
					'woof_tooltip-js',
					'icheck-jquery',
					'woof_front',
					'woof_radio_html_items',
					'woof_checkbox_html_items',
					'woof_select_html_items',
					'woof_mselect_html_items',
					'woof_by_author_html_items',
					'woof_by_instock_html_items',
					'woof_by_onsales_html_items',
					'woof_by_text_html_items',
					'woof_label_html_items',
					'woof_sections_html_items',
					'woof_select_radio_check_html_items',
					'woof_sd_html_items',
					'chosen-drop-down',
					'ion.range-slider',
					'woof-front-builder-boot',
					'woof-front-builder',
					'woof-slideout-js',
					'woof-slideout-init',
					'woof_alasql',
					'woof_turbo_mode'
				];
			}

			if ($remove_quickview_scripts === "__true") {
				$wpc_scripts = [
					'slick',
					'perfect-scrollbar',
					'magnific-popup',
					'woosq-frontend',
				];
			}

			if ($remove_woocommerce_scripts === "__true") {
				$woocommmerce_scripts = [
					'wc-price-slider',
					'wc-jquery-ui-touchpunch',
					'chosen-drop-down',
					'wc-add-to-cart-variation',
					'wp-util',
					'underscore',
					'woocommerce',
					'js-cookie',
					'wc-add-to-cart',
					'prettyPhoto',

					//footer
					'wc-jquery-ui-touchpunch',
					'accounting',
					'wc-price-slider',
				];
			}

			if ($remove_yith_scripts === "__true") {
				$yith_js = [
					'jquery-selectBox',
					'jquery-yith-wcwl',
				];
			}
		}

		$this->dequeue_js_files = array_merge($woof_scripts, $wpc_scripts, $yith_js, $woocommmerce_scripts, $this->dequeue_js_files);

		foreach ($this->dequeue_js_files as $handle) {
			wp_dequeue_script($handle);
		}
	}
}
