<?php

/**
 * Streamit\Utility\Comments\Component class
 *
 * @package streamit
 */

namespace Streamit\Utility\Plugin_Update;

use Streamit\Utility\Component_Interface;

/**
 * Class for managing comments UI.
 *
 * Exposes template tags:
 * * `streamit()->the_comments( array $args = array() )`
 *
 * @link https://wordpress.org/plugins/amp/
 */

class Component implements Component_Interface
{
    /**
     * Gets the unique identifier for the theme component.
     *
     * @return string Component slug.
     */

    protected static $new_version;

    public function get_slug(): string
    {
        return 'plugin_update';
    }

    /**
     * Adds the action and filter hooks to integrate with WordPress.
     */

    public function __construct()
    {
        //NOTE: change this version everytime, when you give update on themeforest
        self::$new_version = "3.3.0";

        if (defined('STREAMIT_EXTENSION_VERSION'))
            $current_plugin_version = STREAMIT_EXTENSION_VERSION;
        else {
            $plugin_data            = get_plugin_data(plugin_dir_path(__FILE__) . '/streamit-extension.php');
            $current_plugin_version = $plugin_data['Version'];
        }

        if (version_compare($current_plugin_version, self::$new_version, '<')) {
            add_action("admin_notices", array($this, "streamit_add_plugin_update_notice"));
            add_filter("pre_set_site_transient_update_plugins", array($this, "update_plugins"));

            add_action("upgrader_process_complete", array($this, 'streamit_regenerate_iqonic_scripts'));
            add_action("upgrader_overwrote_package", array($this, 'streamit_regenerate_iqonic_scripts'));
        }
    }

    public function initialize()
    {
    }

    function streamit_add_plugin_update_notice()
    { ?>
        <div class="notice streamit_plugin_update_notice">
            <div class="streamit-notice-message">
                <h3> <?php esc_html_e('Streamit Extension Plugin Update Notice', STREAMIT_TEXT_DOMAIN); ?> </h3>

                <h4> <?php esc_html_e('It appears that your Streamit Extension Plugin is not up-to-date. Kindly update the plugin now to ensure compatibility and take advantage of performance optimizations.', STREAMIT_TEXT_DOMAIN); ?> </h4>

                <a href="<?php echo admin_url("/plugins.php") ?>">
                    <?php esc_html_e('Click here to initiate the Streamit Extension plugin update.', STREAMIT_TEXT_DOMAIN); ?>
                </a>

                <div class="alert alert-danger" role="alert">
                    <?php esc_html_e("Please remember to create a backup before proceeding with the plugin update as a precautionary measure in case of any issues.", STREAMIT_TEXT_DOMAIN); ?>
                </div>
            </div>
        </div>
<?php

        wp_enqueue_style("streamit-update-plugin-notice", get_stylesheet_uri());
        wp_add_inline_style(
            'streamit-update-plugin-notice',
            '.streamit_plugin_update_notice       { background-color: black; color: white; padding: 1em; }
            .streamit_plugin_update_notice h3     { color: white; }
            .streamit_plugin_update_notice a      { font-weight: 700; margin-top: 1em; margin-bottom: 1em}
            .streamit-notice-message a            {  display: block; }
            .alert-danger                           { color:red; font-weight: 500; } '
        );
    }

    function update_plugins($transient)
    {
        $plugin_slug = "streamit-extensions/streamit-extension.php";

        $transient->response[$plugin_slug] = (object) array(
            'slug'        => 'streamit-extensions',
            'plugin'      => $plugin_slug,
            'url'         => 'https://wordpress.iqonic.design/product/wp/streamit/',
            'package'     => 'https://assets.iqonic.design/wp/plugins/streamit-new/streamit-extensions.zip',
            'new_version' => self::$new_version,
        );

        return $transient;
    }

    function streamit_regenerate_iqonic_scripts()
    {
        if (!class_exists('WP_Filesystem_Direct')) {
            require_once ABSPATH . 'wp-admin/includes/class-wp-filesystem-base.php';
            require_once ABSPATH . 'wp-admin/includes/class-wp-filesystem-direct.php';
        }

        $rm = new \WP_Filesystem_Direct([]);

        $upload_dir = wp_get_upload_dir()['basedir'];
        $css_dir    = $upload_dir . "/streamit/css/";
        $js_dir     = $upload_dir . "/streamit/js/";

        if (file_exists($js_dir))  $rm->delete($js_dir, true, 'd');
        if (file_exists($css_dir)) $rm->delete($css_dir, true, 'd');
    }
}
