<?php

/**
 * Streamit\Utility\Helper\Component class
 *
 * @package streamit
 */

namespace Streamit\Utility\PMP;

use Streamit\Utility\Component_Interface;
use Streamit\Utility\Templating_Component_Interface;
use MemberOrder;

use function Streamit\Utility\streamit;

/**
 * Class for managing comments UI.
 *
 * Exposes template tags:
 * * `streamit()->the_comments( array $args = array() )`
 *
 * @link https://wordpress.org/plugins/amp/
 */

class Component implements Component_Interface, Templating_Component_Interface
{
    /**
     * Gets the unique identifier for the theme component.
     *
     * @return string Component slug.
     */

    public $streamit_options;

    public function get_slug(): string
    {
        return 'pmp';
    }

    public function initialize()
    {
        //adds pmpro-no-access class in body
        add_filter('body_class', array($this, 'my_body_classes'));

        //register custom post types for restriction of MAS videos CPT movies/tvshows/videos
        if (!is_plugin_active('pmpro-cpt/pmpro-cpt.php')) {
            add_action('add_meta_boxes', array($this, 'streamit_pmp_add_restriction_for_cpt'));
        }

        //enqueue custom css an js for pmp
        add_action('wp_enqueue_scripts', array($this, 'streamit_enqueue_custom_pmp_style'), 20);

        //remove pmp join now, lost password links and seperator
        add_filter('pmpro_login_forms_handler_nav', array($this, 'streamit_pmpro_login_form_bottom'), 10, 2);
        add_filter('pmpro_actions_nav_separator', array($this, 'streamit_pmpro_remove_seperator'), 10, 2);

        remove_filter("comments_open", "pmpro_comments_filter", 10, 2);

        add_filter('the_content', array($this, 'streamit_restrict_content_for_non_login'), 15);

        if (function_exists('pmpro_getOption') && pmpro_getOption('showexcerpts') != "1") {
            add_filter('the_excerpt', array($this, 'streamit_restrict_content_for_non_login'), 15);
        }

        //removes the change button from the previously purchased membership plans
        add_filter('pmpro_member_action_links', array($this, 'streamit_pmp_membership_links'));

        //Sets Use of Discount Code to true
        add_filter('pmpro_show_discount_code', array($this, 'streamit_pmp_discount_code'));

        //removes confirm password and confirm email field
        add_filter('pmpro_checkout_confirm_password', array($this, 'streamit_remove_confirm_password_and_email_field'));
        add_filter('pmpro_checkout_confirm_email',    array($this, 'streamit_remove_confirm_password_and_email_field'));

        //registers ajax object scripts
        if (class_exists('Paid_Member_Subscriptions') && is_plugin_active('paid-memberships-pro/paid-memberships-pro.php')) {
            if (is_admin() && current_user_can('administrator')) { //checks admin area and current capability role for permorming import actions
                add_action('admin_enqueue_scripts', array($this, 'streamit_enqueue_custom_pmp_style'));

                //adds import pms plan button
                add_filter('pmpro_membershiplevels_page_action_links', array($this, 'streamit_import_pms_button'));

                //importing pms plans
                add_action('wp_ajax_streamit_import_PMS_plans_and_restriction_content', array($this, 'streamit_import_PMS_plans_and_restriction_content'));
                add_action('wp_ajax_nopriv_streamit_import_PMS_plans_and_restriction_content', array($this, 'streamit_import_PMS_plans_and_restriction_content'));

                add_action('wp_ajax_streamit_import_PMS_member_details', array($this, 'streamit_import_PMS_member_details'));
                add_action('wp_ajax_nopriv_streamit_import_PMS_member_details', array($this, 'streamit_import_PMS_member_details'));
            }
        }

        add_action('wp_ajax_streamit_pmpro_register_wizard', array($this, 'streamit_pmpro_register_wizard'));
        add_action('wp_ajax_nopriv_streamit_pmpro_register_wizard', array($this, 'streamit_pmpro_register_wizard'));

        add_action('template_redirect', array($this, 'streamit_redirect_for_non_login'));

        add_filter('pmpro_checkout_order', array($this, 'streamit_pmp_additional_discount'));

        add_filter('pmpro_not_logged_in_text_filter',   array($this, 'streamit_pmp_formatted_structure'));
        add_filter('pmpro_non_member_text_filter',      array($this, 'streamit_pmp_formatted_structure'));

        add_filter('pmpro_level_cost_text', array($this, 'streamit_pmpro_format_level_cost_text'));

        //PMP Woocmmerce Integration Plugin
        if (is_plugin_active('paid-memberships-pro/paid-memberships-pro.php') && is_plugin_active("pmpro-woocommerce/pmpro-woocommerce.php")) {
            add_action('pmpro_save_membership_level', array($this, 'streamit_add_woocommerce_product'));
        }
    }

    public function template_tags(): array
    {
        return array(
            'streamit_pmpro_register_wizard'      => array($this, 'streamit_pmpro_register_wizard'),
            'streamit_pmpro_woo_payment_gatway'   => array($this, 'streamit_pmpro_woo_payment_gatway'),
            'streamit_register_pmpro_user'        => array($this, 'streamit_register_pmpro_user'),
            'streamit_add_woocommerce_product'    => array($this, 'streamit_add_woocommerce_product'),
        );
    }

    function streamit_pmp_membership_links($links)
    {
        $links['change'] = '';
        return $links;
    }

    function streamit_pmp_additional_discount($order)
    {
        global $streamit_options;
        $additional_discount_amount = (int)$streamit_options['additional_discount_price'];

        if (isset($additional_discount_amount) && $additional_discount_amount > 0) {
            if ($order->InitialPayment >= $additional_discount_amount) {
                $order->InitialPayment = $order->InitialPayment - $additional_discount_amount;
            } else {
                $order->PaymentAmount = $order->PaymentAmount - $additional_discount_amount;
            }
        }
        return $order;
    }

    function streamit_pmp_add_restriction_for_cpt()
    {
        $cpt_for_pmp_plan = apply_filters('streamit_add_cpt_for_pmp', array('movie', 'tv_show', 'video', 'episode'));

        if (class_exists('MasVideos')) {
            foreach ($cpt_for_pmp_plan as $cpt) {
                add_meta_box('pmpro_' . $cpt . '_restrict', __('Require Membership', ''), 'pmpro_page_meta', $cpt, 'side', 'high');
            }
        }

        if (is_plugin_active("pmpro-addon-packages/pmpro-addon-packages.php")) {
            foreach ($cpt_for_pmp_plan as $cpt) {
                add_meta_box('pmproap_post_meta', __('PMPro Addon Package Settings', 'pmpro-addon-packages'), 'pmproap_post_meta', $cpt, 'normal');
            }
        }
    }

    function streamit_enqueue_custom_pmp_style()
    {
        //Enqueuq Only For PMP pages
        $pmpro_pages = [
            pmpro_getOption('levels_page_id'),
            pmpro_getOption('checkout_page_id'),
            pmpro_getOption('confirmation_page_id'),
            pmpro_getOption('account_page_id'),
            pmpro_getOption('billing_page_id'),
            pmpro_getOption('cancel_page_id'),
            pmpro_getOption('invoice_page_id'),
            pmpro_getOption('login_page_id'),
            pmpro_getOption('member_profile_edit_page_id'),
        ];

        if (in_array(get_queried_object_id(), $pmpro_pages)) {
            wp_enqueue_style('streamit-pmp', get_template_directory_uri() . "/assets/css/streamit-pmp.min.css", array(), streamit()->get_version());
            wp_enqueue_script('streamit-pmp', get_template_directory_uri() . "/assets/js/streamit-pmp.min.js", array('jquery'), streamit()->get_version(), [true, 'async']);
            wp_localize_script('streamit-pmp', 'streamit_pmpro_object', array(
                'ajaxurl' => admin_url('admin-ajax.php'),
            ));
        }
    }

    public function my_body_classes($classes)
    {
        if (function_exists('pmpro_has_membership_access') && pmpro_has_membership_access(get_the_ID()) === false) {
            $classes[] = 'pmpro-no-access';
        }

        if (function_exists('pmpro_getOption') && empty(pmpro_getOption("showexcerpts"))) {
            $classes[] = 'pmpro-no-excerpt';
        }

        return $classes;
    }

    public function streamit_pmpro_register_wizard()
    {
        $current_page_id = get_queried_object_id();

        $membership_level_page_id = pmpro_getOption('levels_page_id');
        $membership_level_page_link = !empty($membership_level_page_id) ? get_the_permalink($membership_level_page_id) : '';

        $confirmation_page_id = pmpro_getOption('confirmation_page_id');
        if ($current_page_id == $confirmation_page_id) {
            $checkout_page_id = pmpro_getOption('checkout_page_id');
            $checkout_page_link = !empty($checkout_page_id) ? get_the_permalink($checkout_page_id) : '';
        } else {
            $checkout_page_link = '';
        }

        $steps = array(
            array(
                'name' => esc_html__('Select Plan', 'streamit'),
                'id' => 'streamit-pmpro_select_plan',
                'link' => $membership_level_page_link,
            ),
            array(
                'name' => esc_html__('Account Details', 'streamit'),
                'id' => 'streamit-pmpro_register',
                'link' => $checkout_page_link,
            ),
            array(
                'name' => esc_html__('Order Summary', 'streamit'),
                'id' => 'streamit-pmpro_checkout_success',
            )
        ); ?>

        <div class="streamit-page-header">
            <ul class="streamit-page-items">
                <?php foreach ($steps as $key => $item) {
                    $href = '';
                    if (isset($item['link']) && !empty($item['link'])) {
                        $href = 'href=' . $item['link'] . '';
                    } ?>
                    <li class="streamit-page-item" id="<?php echo esc_attr($item['id']) ?>">
                        <span class="streamit-pre-heading"> <?php echo esc_html($key + 1) ?> </span>
                        <a class="streamit-page-link" <?php echo esc_attr($href); ?>>
                            <?php echo esc_html($item['name']); ?>
                        </a>
                    </li>
                <?php } ?>
            </ul>
        </div>
<?php
        if (is_ajax()) {
            die;
        }
    }

    function streamit_pmpro_login_form_bottom($links, $pmpro_form)
    {
        // remove the "Lost password" link
        if ($pmpro_form != 'lost_password') {
            $links['lost_password'] = '';
        }
        // remove the "register" link
        if ($pmpro_form == "login") {
            $links['register'] = '';
        }
        return $links;
    }

    function streamit_pmpro_remove_seperator($seperator)
    {
        $seperator = '';
        return $seperator;
    }

    function streamit_pmp_discount_code($show)
    {
        return true;
    }

    function streamit_remove_confirm_password_and_email_field()
    {
        return false;
    }

    function streamit_import_pms_button($pmpro_membershiplevels_page_action_links)
    {
        $pmpro_membershiplevels_page_action_links['import-pms-plans'] = array(
            'url' => '#',
            'name' => esc_html__('Import PMS Plans', 'streamit'),
            'icon' => 'plus import_pms_plans'
        );

        $pmpro_membershiplevels_page_action_links['import-pms-members-info'] = array(
            'url' => '#',
            'name' => esc_html__('Import PMS Member\'s Detail', 'streamit'),
            'icon' => 'plus import_pms_members_info'
        );

        return $pmpro_membershiplevels_page_action_links;
    }

    function streamit_import_PMS_plans_and_restriction_content()
    {
        $this->streamit_import_PMS_plans();
        echo '<br>';
        $this->streamit_import_restrict_content();
        die;
    }

    function streamit_import_PMS_plans()
    {
        global $wpdb;

        $subscription_plans = pms_get_subscription_plans(false); // get PMS subscription plans list
        $pmpro_levels = pmpro_sort_levels_by_order(pmpro_getAllLevels(true, false)); // get PMP subscription plans list and append at last

        //return if there is no PMS plans
        if (empty($subscription_plans)) {
            echo esc_html__('No PMS plans to import', 'streamit');
            return;
        }

        foreach ($subscription_plans as $key => $pms_plan) {
            if ($pmpro_levels[$pms_plan->id]->id == $pms_plan->id) continue; //skips current loop if plan id is already present

            $import_pmp_plans = array(); //empty the array for next loop

            $expiration_number = (!empty($pms_plan->duration)) ? $pms_plan->duration : 0;
            $expiration_period = (!empty($pms_plan->duration_unit)) ? ucfirst($pms_plan->duration_unit) : 0;
            $allow_signup = ($pms_plan->status == 'active') ? 1 : 0;

            //==========WARNING: DO NOT CHANGE THE ARRAY ORDER============
            $import_pmp_plans = array(
                'id' => $pms_plan->id,                     // PMS Plan ID
                'name' => $pms_plan->name,                 // PMS Plan Name
                'description' => $pms_plan->description,   // PMS Plan Description
                'confirmation' => '',                      // Featured not provided by PMS, so default empty
                'initial_payment' => $pms_plan->price,     // PMS Plan Price
                'billing_amount' => $pms_plan->price,      // PMS Plan Price
                'cycle_number' => $expiration_number,      // PMS expiry number
                'cycle_period' => $expiration_period,      // PMS expiry period
                'billing_limit' => 0,                      // Featured not provided by PMS, so default 0
                'trial_amount' => 0,                       // Featured not provided by PMS, so default 0
                'trial_limit' => 0,                        // Featured not provided by PMS, so default 0
                'expiration_number' => 0,                  // Featured not provided by PMS, so default 0
                'expiration_period' => 0,                  // Featured not provided by PMS, so default 0
                'allow_signups' => $allow_signup           // PMS is plan active
            );

            pmpro_insert_or_replace(
                $wpdb->pmpro_membership_levels,
                $import_pmp_plans, //our newly created array to import PMS plans to PMP
                array(
                    '%d',        //id
                    '%s',        //name
                    '%s',        //description
                    '%s',        //confirmation
                    '%f',        //initial_payment
                    '%f',        //billing_amount
                    '%d',        //cycle_number
                    '%s',        //cycle_period
                    '%d',        //billing_limit
                    '%f',        //trial_amount
                    '%d',        //trial_limit
                    '%d',        //expiration_number
                    '%s',        //expiration_period
                    '%d',        //allow_signups
                )
            );
        }
        echo esc_html__('PMS Plans Successfully Imported.', 'streamit');
        return;
    }

    function streamit_import_restrict_content()
    {
        global $wpdb;

        $post_types = apply_filters('streamit_import_pms_restrict_post_type', array('movie', 'tv_show', 'episode', 'video', 'person', 'post', 'page'));
        $args = array(
            'post_type' => $post_types,
            'post_status' => 'all',
            'posts_per_page' => -1,
            'meta_query' => array(
                array(
                    'key' => 'pms-content-restrict-subscription-plan',
                )
            ),
            'orderby' => 'meta_value',
            'order' => 'ASC',
        );

        query_posts($args);
        if (have_posts()) :
            $sql_values = '';
            while (have_posts()) : the_post();
                $post_id = get_the_ID();
                $membership_plans_id = get_post_meta($post_id, 'pms-content-restrict-subscription-plan');

                $sql = "SELECT `membership_id`, `page_id` FROM $wpdb->pmpro_memberships_pages
                            INNER JOIN $wpdb->pmpro_membership_levels
                            ON $wpdb->pmpro_membership_levels.id = $wpdb->pmpro_memberships_pages.membership_id
                            WHERE page_id = %d";

                $membership_levels_names = $wpdb->get_results($wpdb->prepare($sql, [$post_id]), ARRAY_A);


                if (!empty($membership_plans_id) && is_array($membership_plans_id)) {
                    foreach ($membership_plans_id as $key => $membership_plan_id) {
                        if (isset($membership_levels_names[$key]) && $membership_levels_names[$key]['membership_id'] !== $membership_plan_id) {
                            $sql_values .= '(' . intval($membership_plan_id) . ',' . intval($post_id) . '),';
                        } else {
                            $sql_values .= '(' . intval($membership_plan_id) . ',' . intval($post_id) . '),';
                        }
                    }
                }
            endwhile;
        endif;
        wp_reset_query();

        if (isset($sql_values) && !empty($sql_values)) {
            $sql_values = rtrim($sql_values, ','); // remove last comma
            $sql = "INSERT IGNORE INTO {$wpdb->pmpro_memberships_pages} (membership_id, page_id)
                VALUES $sql_values";

            $result = $wpdb->query($sql);

            if (!$result) {
                echo esc_html__("Posts Restricted Content Already Imported", 'streamit');
                return;
            }
        }
        echo esc_html__('Sucessfully Imported Posts Rescticted Content.', 'streamit');
        return;
    }

    function streamit_import_PMS_member_details()
    {
        $this->streamit_import_PMS_members_data();
        echo '<br>';
        $this->streamit_import_pms_payments();
        die;
    }

    function streamit_import_PMS_members_data()
    {
        global $wpdb;

        $members = $wpdb->get_results("SELECT * FROM {$wpdb->prefix}pms_member_subscriptions");
        if (empty($members)) {
            echo esc_html__('No PMS Members Data to Import.', 'streamit');
            return;
        }

        $user_ids = $wpdb->get_col("SELECT user_id FROM $wpdb->pmpro_memberships_users");
        if (!is_array(($user_ids))) die;

        $sql_values = '';

        foreach ($members as $member) {
            if (in_array($member->user_id, $user_ids)) continue;

            $user_id = $member->user_id;
            $member_id = $member->subscription_plan_id;
            $code_id = 0;
            $initial_payment = $billing_amount = $member->billing_amount;
            $cycle_number = 0;
            $cycle_period = '';
            $billing_limit = 0;
            $trial_amount = 0;
            $trial_limit = 0;
            $startdate = $member->start_date;
            $enddate = $member->expiration_date;

            $sql_values .= '(' .
                "'" . $user_id . "'," .           //user id
                "'" . $member_id . "'," .        //subscription_plan_id
                "'" . $code_id . "'," .          //discount code used or not value 0 or 1
                "'" . $initial_payment . "'," .  //plan amount
                "'" . $billing_amount . "'," .   //
                "'" . $cycle_number . "'," .
                "'" . $cycle_period . "'," .
                "'" . $billing_limit . "'," .
                "'" . $trial_amount . "'," .
                "'" . $trial_limit . "'," .
                "'" . $startdate . "'," .
                "'" . $enddate . "'" .
                '),';
        }

        if (isset($sql_values) && !empty($sql_values)) {
            $sql_values = rtrim($sql_values, ','); // remove last comma

            $sql = "INSERT INTO {$wpdb->pmpro_memberships_users}
            (`user_id`, `membership_id`, `code_id`, `initial_payment`, `billing_amount`, `cycle_number`, `cycle_period`, `billing_limit`, `trial_amount`, `trial_limit`, `startdate`, `enddate`)
            VALUES $sql_values;";

            $result = $wpdb->query($sql);

            if (!$result) {
                echo esc_html__("Members Data Already Imported.", 'streamit');
                return;
            }

            echo esc_html__('Sucessfully Imported Members Data.', 'streamit');
            return;
        } else {
            echo esc_html__("Members Data Already Imported.", 'streamit');
            return;
        }
    }

    function streamit_import_pms_payments()
    {
        global $wpdb;

        $pms_payments = $wpdb->get_results("SELECT * FROM {$wpdb->prefix}pms_payments");
        if (empty($pms_payments)) {
            echo esc_html__("No Payments Data To Import.", 'streamit');
            return;
        }

        $user_ids = $wpdb->get_col("SELECT user_id FROM $wpdb->pmpro_membership_orders");
        $sql_values = '';
        $order = new MemberOrder();

        foreach ($pms_payments as $pms_payment) {

            if (in_array($pms_payment->user_id, $user_ids)) continue;

            $time = time();
            $timestamp = date("Y-m-d H:i:s", $time);

            $code = $order->getRandomCode();
            $user_id = $pms_payment->user_id;
            $membership_id = $pms_payment->subscription_plan_id;
            $status = $pms_payment->status;
            $subtotal = $total = $pms_payment->amount;
            $timestamp = $timestamp;

            $sql_values .= '(' .
                "'" . $code . "'," .          //random payment code
                "'" . $user_id . "'," .       //user id
                "'" . $membership_id . "'," . //subscription_plan_id
                "'" . $subtotal . "'," .      //purchased plan amount
                "'" . $total . "'," .         //purchased plan amount
                "'" . $status . "'," .        //payment status
                "'" . $timestamp . "'" .     //TIMESTAMP
                '),';
        }

        if (isset($sql_values) && !empty($sql_values)) {
            $sql_values = rtrim($sql_values, ','); // remove last comma

            $sql = "INSERT INTO $wpdb->pmpro_membership_orders
            (`code`, `user_id`, `membership_id`, `subtotal`, `total`, `status`, `timestamp`)
            VALUES $sql_values";

            $result = $wpdb->query($sql);

            if (!$result) {
                echo esc_html__("Payments Data Already Imported.", 'streamit');
            }

            echo esc_html__('Sucessfully Imported Payments Data.', 'streamit');
            return;
        } else {
            echo esc_html__("Payments Data Already Imported.", 'streamit');
            return;
        }
    }

    function streamit_redirect_for_non_login()
    {
        global $streamit_options;
        
        $by_pass = apply_filters("streamit_restriction_bypass", false);
        if($by_pass) return;

        if (!is_user_logged_in()) {
            if (isset($streamit_options['restrict_non_logged_in']) && $streamit_options['restrict_non_logged_in'] == "yes") {
                if (isset($streamit_options['streamit_redirect_page_for_non_login']) && !empty($streamit_options['streamit_redirect_page_for_non_login'])) {
                    if (isset($streamit_options['streamit_noredirect_page_for_non_login']) && !empty($streamit_options['streamit_noredirect_page_for_non_login'])) {

                        $noredirect_pages = $streamit_options['streamit_noredirect_page_for_non_login'];
                        $noredirect_pages = apply_filters("streamit_noredirect_pages", $noredirect_pages);

                        $page_id = get_queried_object_id();

                        foreach ($noredirect_pages as $noredirect_page) {
                            if (is_page($noredirect_page)) return;
                            if ($page_id == $noredirect_page) return;
                        }
                    }

                    //return if current pages are PMP Pricing Plan/Checkout/Confirmation/Login Pages
                    if (
                        is_page($streamit_options['streamit_redirect_page_for_non_login']) ||
                        is_page(pmpro_getOption('levels_page_id')) ||
                        is_page(pmpro_getOption('checkout_page_id')) ||
                        is_page(pmpro_getOption('confirmation_page_id')) ||
                        is_page(pmpro_getOption('login_page_id'))
                    ) return;

                    wp_redirect(get_page_link($streamit_options['streamit_redirect_page_for_non_login']));
                    exit;
                }
            }
        }
    }

    function streamit_restrict_content_for_non_login($content)
    {
        global $post, $wpdb, $streamit_options;

        $streamit_restrict_content_for_non_login = apply_filters("streamit_restrict_content_for_non_login", true);

        if ($streamit_restrict_content_for_non_login && !is_user_logged_in()) {
            $logged_out_cpt = apply_filters('streamit_add_pmpro_logged_out_cpt', array('movie', 'tv_show', 'video', 'person', 'post'));

            //return if current pages are PMP Pricing Plan/Checkout/Confirmation/Login Pages
            if (is_page(pmpro_getOption('levels_page_id')) || is_page(pmpro_getOption('checkout_page_id')) || is_page(pmpro_getOption('confirmation_page_id'))) return $content;

            if ((is_singular($logged_out_cpt) || is_home() || is_tag() || is_category() || in_array(get_post_type(), $logged_out_cpt)) && $streamit_options['restrict_non_logged_in'] == "yes") {
                $sql = "SELECT `id` FROM $wpdb->pmpro_memberships_pages
                INNER JOIN $wpdb->pmpro_membership_levels
                ON $wpdb->pmpro_membership_levels.id = $wpdb->pmpro_memberships_pages.membership_id
                WHERE page_id = %d";

                $membership_levels_names = $wpdb->get_results($wpdb->prepare($sql, [$post->ID]), ARRAY_A);

                if (!empty($membership_levels_names)) {
                    foreach ($membership_levels_names as $membership_names) {
                        $post_membership_levels_id[] = $membership_names['id'];
                    }
                } else {
                    $post_membership_levels_id = array();
                }

                $content = pmpro_get_no_access_message('', $post_membership_levels_id);
            }
        }
        return $content;
    }

    function streamit_pmp_formatted_structure($content)
    {
        $html = '';

        if (get_post_type() == "tv_show") return $html;

        $html = '
            <div class="streamit_content_restriction">
                <div class="streamit_restriction_box">
                ' . $content . '
                </div>
            </div>';

        return $html;
    }

    function streamit_pmpro_format_level_cost_text($text)
    {
        $text = str_replace(array("every", "per"), '/', $text);
        $text = str_replace(array("now", ' .'), '', $text);

        return $text;
    }

    /**
     * Creates an order for PMP Woocommerce Payment Gateway and automatically activates the free account of PMP
     * 
     * @param plan_id The PMPRO plan_id to buy.
     * 
     * @param product_id The product_id of the associated PMPRO plan id.
     * 
     * @param redirect_url The `redirect_url` parameter is the URL where the user will be redirected
     * after the function completes its execution. It can be a checkout payment URL or the account
     * endpoint URL, depending on whether the user is logged in or not.
     * 
     */

    function streamit_pmpro_woo_payment_gatway($plan_id = NULL, $product_id = NULL, $redirect_url = NULL)
    {
        $user_id            = get_current_user_id();
        $is_user_logged_in  = is_user_logged_in();
        $generate_new_order = true;

        if ($plan_id == NULL) {
            $plan_id = isset($_REQUEST['plan_id']) ? $_REQUEST['plan_id'] : NULL;
        }

        if ($product_id == NULL) {
            $product_id = isset($_REQUEST['product_id']) ? $_REQUEST['product_id'] : NULL;
        }

        if (($plan_id === NULL) && ($product_id === NULL)) die;

        //get product
        $obj_product = wc_get_product($product_id);
        if (!$obj_product) {
            $obj_product = wc_get_product_object("product", $product_id);
        }

        $is_product_free = $obj_product->get_regular_price() == "0";

        //return order checkout url, if exists in the cookie
        if (isset($_COOKIE['pmpro_checkout_payment_url'][$user_id][$plan_id]) && !empty($_COOKIE['pmpro_checkout_payment_url'][$user_id][$plan_id])) {

            $pmpro_checkout_payment_url = $_COOKIE['pmpro_checkout_payment_url'][$user_id][$plan_id];

            //check if $pmpro_checkout_payment_url order is completed, than create a new order
            $query_string = parse_url($pmpro_checkout_payment_url);

            // GET THE ORDER ID FROM THE URL
            $path       = $query_string['path'];
            $segments   = explode('/', $path);
            $segments   = array_filter($segments); //removes empty values from the array
            $order_id   = end($segments);

            // check if the order status is completed, if completed, than generate new order, for selected plan
            $order          = new \WC_Order($order_id);
            $order_status   = $order->get_status();

            $generate_new_order = ($order_status == "completed" || $order_status != "pending") ? true : false;

            // $generate_new_order == false, return previous created order url
            if (!$generate_new_order)
                $redirect_url = $_COOKIE['pmpro_checkout_payment_url'][$user_id][$plan_id];
        }

        if (!$generate_new_order) {
            $redirect_url = $_COOKIE['pmpro_checkout_payment_url'][$user_id][$plan_id];
        } else {
            //create order
            $order      = wc_create_order();
            $order_id   = $order->get_id();

            if (class_exists('WC_Gateway_Cinetpay')) {
                global $woocommerce;

                WC()->cart->add_to_cart($product_id, 1);
                $redirect_url = $woocommerce->cart->get_checkout_url();

                return $redirect_url;
            }

            $order->add_product($obj_product);
            $order->set_customer_id(get_current_user_id());
            $order->calculate_totals();
            $order->update_status("pending", 'Streamit PMP Plan Order Placed', true);

            if ($is_product_free && function_exists('pmprowoo_order_autocomplete')) {
                pmprowoo_order_autocomplete($order_id);
            }

            $redirect_url = $order->get_checkout_payment_url();

            //sets cookie for 30 Minutes, so everytime user click on plan, new order is not generated
            setcookie("pmpro_checkout_payment_url[$user_id][$plan_id]", $redirect_url, time() + 1800);
        }

        if (!wp_doing_ajax()) {
            return $redirect_url;
        }

        //must pass only checkout url, if changes, than change in the js also
        wp_send_json_success($redirect_url);
        die;
    }

    /**
     * Registers a new user for a free membership level
     * in the PMPro plugin for WordPress.
     * 
     * @param new_user_id The new_user_id parameter is the ID of the user that needs to be registered
     * for a membership level.
     * 
     * @return (bool) true if user is regsitered, otherwise false.
     * 
     */

    function streamit_register_pmpro_user($new_user_id)
    {
        global $wpdb;

        if (pmpro_getMembershipLevelForUser($new_user_id) !== false) return false;

        $levels = pmpro_getAllLevels(false, true);
        foreach ($levels as $level) {
            $is_free = pmpro_isLevelFree($level);
            if ($is_free) {
                $free_level = $level;
                break;
            }
        }

        if (isset($free_level)) {
            $sql = $wpdb->prepare(
                "
				INSERT INTO {$wpdb->pmpro_memberships_users}
				(`user_id`, `membership_id`, `code_id`, `initial_payment`, `billing_amount`, `cycle_number`, `cycle_period`, `billing_limit`, `trial_amount`, `trial_limit`, `startdate`, `enddate`)
				VALUES ( %d, %d, %d, %s, %s, %d, %s, %d, %s, %d, %s, %s )",
                $new_user_id,
                $free_level->id,
                0,
                $free_level->initial_payment,
                $free_level->billing_amount,
                $free_level->cycle_number,
                $free_level->cycle_period,
                $free_level->billing_limit,
                $free_level->trial_amount,
                $free_level->trial_limit,
                current_time('mysql'),
                '0000-00-00 00:00:00',
            );

            return $wpdb->query($sql);
        }
    }

    function streamit_add_woocommerce_product($plan_id)
    {
        $is_product_created = $this->streamit_create_woocommerce_product($plan_id);

        if ($is_product_created) {
            $meta_key = '_plan_product_id';
            add_pmpro_membership_level_meta($plan_id, $meta_key, $is_product_created, true);
        }
    }

    function streamit_create_woocommerce_product($plan_id)
    {
        $pmpro_level = pmpro_getLevel($plan_id);

        if (empty($pmpro_level)) return false;

        $product_name = 'PMP ' . $pmpro_level->name;

        $product_exists = get_page_by_path($product_name, OBJECT, 'product');

        if ($product_exists) {
            update_post_meta($product_exists->ID, '_price', $pmpro_level->initial_payment);
            update_post_meta($product_exists->ID, '_regular_price', $pmpro_level->initial_payment);

            update_post_meta($product_exists->ID, '_visibility', 'hidden');
            $terms = array('exclude-from-search', 'exclude-from-catalog');
            wp_set_post_terms($product_exists->ID, $terms, 'product_visibility', false);

            return $product_exists->ID;
        }

        $product = array(
            'post_title'    => $product_name,
            'post_type'     => 'product',
            'post_status'   => 'publish',
        );

        $product_id = wp_insert_post($product);

        update_post_meta($product_id, '_sold_individually', 'yes');                     //sets sold individually
        update_post_meta($product_id, '_price', $pmpro_level->initial_payment);
        update_post_meta($product_id, '_regular_price', $pmpro_level->initial_payment); //sets product price

        update_post_meta($product_id, '_visibility', 'hidden');                         //sets visibility hidden
        $terms = array('exclude-from-search', 'exclude-from-catalog');
        wp_set_post_terms($product_id, $terms, 'product_visibility', false);

        update_post_meta($product_id, '_membership_product_level', $pmpro_level->id);   //sets membership id
        update_post_meta($product_id, '_membership_product_autocomplete', true);        //sets autocomplete for product

        wc_update_product_stock($product_id, 0, 'set');
        return $product_id;
    }
} ?>