<?php

/**
 * Streamit\Utility\Import\Component class
 *
 * @package streamit
 */

namespace Streamit\Utility\Import;

use Streamit\Utility\Component_Interface;

class Component implements Component_Interface
{
	/**
	 * Gets the unique identifier for the theme component.
	 *
	 * @return string Component slug.
	 */

	public function get_slug(): string
	{
		return 'import';
	}

	/**
	 * Adds the action and filter hooks to integrate with WordPress.
	 */

	public function initialize()
	{
		add_filter('merlin_generate_child_functions_php', 	array($this, 'streamit_generate_child_functions_php'), 10, 2);
		add_filter('merlin_generate_child_style_css', 		array($this, 'streamit_generate_child_style_css'), 10, 5);
		add_filter('merlin_import_files', 					array($this, 'streamit_import_files'));
		add_action('merlin_after_all_import', 				array($this, 'streamit_after_import_setup'));
		add_filter('merlin_generate_child_screenshot', function () {
			return trailingslashit(get_template_directory()) . 'screenshot.png';
		});
	}

	function streamit_generate_child_functions_php($output, $slug)
	{

		$slug_no_hyphens = strtolower(preg_replace('#[^a-zA-Z]#', '', $slug));

		$output = "
		<?php
		/**
		 * Theme functions and definitions.
		 */
		add_action( 'wp_enqueue_scripts', '{$slug_no_hyphens}_enqueue_styles' ,99);

		function {$slug_no_hyphens}_enqueue_styles() {
				
			wp_enqueue_style( 'parent-style', get_stylesheet_directory_uri() . '/style.css'); 
			wp_enqueue_style( 'child-style',get_stylesheet_directory_uri() . '/style.css');
		}

		/**
		 * Set up My Child Theme's textdomain.
		*
		* Declare textdomain for this child theme.
		* Translations can be added to the /languages/ directory.
		*/
		function {$slug_no_hyphens}_child_theme_setup() {
			load_child_theme_textdomain( 'streamit', get_stylesheet_directory() . '/languages' );
		}
		add_action( 'after_setup_theme', '{$slug_no_hyphens}_child_theme_setup' );
	";

		// Let's remove the tabs so that it displays nicely.
		$output = trim(preg_replace('/\t+/', '', $output));

		// Filterable return.
		return $output;
	}

	public function streamit_generate_child_style_css($output, $slug, $author, $parent, $version)
	{

		$output = "
			/**
			* Theme Name: {$parent} Child
			* Description: This is a child theme of {$parent}, generated by iQonic Design.
			* Author: {$author}
			* Template: {$slug}
			* Version: {$version}
			*/\n
		";

		// Let's remove the tabs so that it displays nicely.
		$output = trim(preg_replace('/\t+/', '', $output));

		return  $output;
	}

	function streamit_import_files()
	{
		return array(
			array(
				'import_file_name'             => esc_html__('All Content', 'streamit'),
				'local_import_file'            => trailingslashit(get_template_directory()) . '/inc/Import/Demo/streamit-content.xml',
				'local_import_widget_file'     => trailingslashit(get_template_directory()) . '/inc/Import/Demo/streamit-widget.wie',
				'local_import_customizer_file' => trailingslashit(get_template_directory()) . '/inc/Import/Demo/streamit-export.dat',
				'local_import_redux'           => array(
					array(
						'file_path'   => trailingslashit(get_template_directory()) . 'inc/Import/Demo/streamit_redux.json',
						'option_name' => 'streamit_options',
					),
				),
				'import_preview_image_url'     => get_template_directory_uri() . '/screenshot.png',
				'import_notice' 			   => esc_html__('DEMO IMPORT REQUIREMENTS: Memory Limit of 128 MB and max execution time (php time limit) of 300 seconds. ', 'streamit') . '</br></br>' . esc_html__('Based on your INTERNET SPEED it could take 5 to 25 minutes. ', 'streamit'),
				'preview_url'                  => 'https://wordpress.iqonic.design/product/wp/streamit',
			),
		);
	}

	function streamit_after_import_setup()
	{
		global $wp_filesystem;

		$content = '';

		// Assign menus to their locations.
		$menus     = wp_get_nav_menus();
		$locations = get_theme_mod('nav_menu_locations');

		if ($menus) {
			foreach ($menus as $menu) {
				if ($menu->name == 'Main Menu') {
					$locations['top'] = $menu->term_id;
				}
			}
		}
		set_theme_mod('nav_menu_locations', $locations); // set menus to locations 

		//reading settings default static pages
		$front_page_id = get_page_by_path('OTT Home Page');
		$blog_page_id  = get_page_by_path('Blog');

		update_option('show_on_front', 'page');
		update_option('page_on_front', $front_page_id->ID);
		update_option('page_for_posts', $blog_page_id->ID);

		//updates quick view icon
		update_option('woosq_button_type', 'link');
		update_option('woosq_button_position', '0');

		//updates WP Ulike Icon
		// $wp_ulike_setting = get_option('wp_ulike_settings');
		// $wp_ulike_setting['posts_group']['enable_auto_display'] = '0';
		// update_option('wp_ulike_settings', $wp_ulike_setting);

		//change woocommerce default shop page
		$id = get_page_by_path('shop-page');
		$id = $id->ID;
		update_option('woocommerce_shop_page_id', $id);

		//import elementor settings
		$import_file =  trailingslashit(get_template_directory()) . 'inc/Import/Demo/elementor-site-settings.json';
		if (file_exists($import_file)) {
			if (empty($wp_filesystem)) {
				require_once(ABSPATH . '/wp-admin/includes/file.php');
				WP_Filesystem();

				$content = $wp_filesystem->get_contents($import_file);
			}
		}

		if (!empty($content)) {
			$default_post_id = get_option('elementor_active_kit');
			update_post_meta($default_post_id, '_elementor_page_settings', json_decode($content, true));
		}

		//Elementor post-types settings for edit with elementor option
		$enable_edit_with_elementor = [
			"post",
			"page",
			"iqonic_hf_layout",
		];
		update_option('elementor_cpt_support', $enable_edit_with_elementor);

		// save woof setting
		$woof_setting_file =  trailingslashit(get_template_directory()) . 'inc/Import/Demo/streamit-woof-setting.json';
		if (file_exists($woof_setting_file)) {
			$content =  $wp_filesystem->get_contents($woof_setting_file);
			if (!empty($content)) {
				$woof_options = json_decode($content, true);
				foreach ($woof_options as $option_name => $option_data) {
					if (is_serialized($option_data)) {
						$option_data = unserialize($option_data);
					}
					update_option($option_name, $option_data);
				}
			}
		}

		//REGISTERS 3 DEFAULT PLAN FOR PMP
		$this->register_default_pmp_plans();

		//GENERATE PMPRO PAGES
		$this->generate_default_pmpro_pages();

		// Update Menu Item For Demo Import
		$menu_item = get_posts(array(
			'post_type' => array('nav_menu_item', 'movie', 'tv_show', 'episode', 'video', 'product'),
			'post_status' => 'publish',
			'numberposts' => -1,
		));

		foreach ($menu_item as $key => $value) {
			wp_update_post(
				array(
					'ID' 			=> $value->ID,
					'post_content' 	=> $value->post_content,
				)
			);
		}

		// remove default post
		wp_delete_post(1, true);
	}

	function register_default_pmp_plans()
	{
		global $wpdb;

		if (!function_exists('is_plugin_active'))  require ABSPATH . '/wp-admin/includes/plugin.php';

		if (function_exists('is_plugin_active')) {
			if (is_plugin_active('paid-memberships-pro/paid-memberships-pro.php') && function_exists("pmpro_sort_levels_by_order")) {

				$pmpro_levels = pmpro_sort_levels_by_order(pmpro_getAllLevels(true, false)); // get PMP subscription plans list and append at last
				$default_pmpro_levels = [
					'8249' =>
					[
						'id' 				=> 8249,
						'name' 				=> esc_html__("Free", "streamit"),
						'description' 		=> "",
						'confirmation'		=> "",
						'initial_payment' 	=> 0,
						'billing_amount' 	=> 0,
						'cycle_number' 		=> 0,
						'cycle_period' 		=> esc_html__("Day", "streamit"),
					],
					'8267' =>
					[
						'id' 				=> 8267,
						'name' 				=> esc_html__("Basic", "streamit"),
						'description' 		=> "",
						'confirmation'		=> "",
						'initial_payment' 	=> 19,
						'billing_amount' 	=> 19,
						'cycle_number' 		=> 1,
						'cycle_period' 		=> esc_html__("Month", "streamit"),
					],
					'8269' =>
					[
						'id' 				=> 8269,
						'name' 				=> esc_html__("Premium", "streamit"),
						'description' 		=> esc_html__("Enjoy all the benefits all Streamit", "streamit"),
						'confirmation'		=> "",
						'initial_payment' 	=> 39,
						'billing_amount' 	=> 39,
						'cycle_number' 		=> 3,
						'cycle_period' 		=> esc_html__("Month", "streamit"),
					],
				];

				foreach ($default_pmpro_levels as $key => $pmp_plan) {
					if ($default_pmpro_levels[$key]['id'] == $pmpro_levels[$key]->id) continue; //skips current loop if plan id is already present

					$register_default_pmp_plans = array(); //empty the array for next loop

					//==========WARNING: DO NOT CHANGE THE ARRAY ORDER============
					$register_default_pmp_plans = array(
						'id' 				=> $pmp_plan['id'],
						'name' 				=> $pmp_plan['name'],
						'description' 		=> $pmp_plan['description'],
						'confirmation' 		=> $pmp_plan['confirmation'],
						'initial_payment' 	=> $pmp_plan['initial_payment'],
						'billing_amount' 	=> $pmp_plan['billing_amount'],
						'cycle_number' 		=> $pmp_plan['cycle_number'],
						'cycle_period' 		=> $pmp_plan['cycle_period'],
						'billing_limit' 	=> 0,
						'trial_amount' 		=> 0,
						'trial_limit' 		=> 0,
						'expiration_number' => 0,
						'expiration_period' => 0,
						'allow_signups' 	=> 1
					);

					pmpro_insert_or_replace(
						$wpdb->pmpro_membership_levels,
						$register_default_pmp_plans, //our newly created array to register default PMP plans
						array(
							'%d',        //id
							'%s',        //name
							'%s',        //description
							'%s',        //confirmation
							'%f',        //initial_payment
							'%f',        //billing_amount
							'%d',        //cycle_number
							'%s',        //cycle_period
							'%d',        //billing_limit
							'%f',        //trial_amount
							'%d',        //trial_limit
							'%d',        //expiration_number
							'%s',        //expiration_period
							'%d',        //allow_signups
						)
					);
				}
			}
		}
	}

	function generate_default_pmpro_pages()
	{
		global $wpdb;

		if (function_exists('pmpro_setOption')) {
			$generate_pmpro_pages = [
				'account' => [
					esc_html__('Membership Account', 'streamit'),
					'membership-account'
				],
				'billing' => [
					esc_html__('Membership Billing', 'streamit'),
					'membership-billing'
				],
				'cancel' => [
					esc_html__('Membership Cancel', 'streamit'),
					'membership-cancel',
				],
				'checkout' => [
					esc_html__('Membership Checkout', 'streamit'),
					'membership-checkout',
				],
				'confirmation' => [
					esc_html__('Membership Confirmation', 'streamit'),
					'membership-confirmation',
				],
				'invoice' => [
					esc_html__('Membership Invoice', 'streamit'),
					'membership-invoice',
				],
				'levels' => [
					esc_html__('Membership Levels', 'streamit'),
					'pricing-plan',
				],
				'login' => [
					esc_html__('Log In', 'streamit'),
					'login',
				],
				'member_profile_edit' => [
					esc_html__('Your Profile', 'streamit'),
					'your-profile',
				],
			];

			$pages = array();

			foreach ($generate_pmpro_pages as $key => $pmpro_pages) {
				$pmpro_page = $pmpro_pages[1];

				if (!empty($pmpro_page)) {
					$sql = "
				SELECT ID, post_name, post_parent, post_type
				FROM $wpdb->posts
				WHERE post_name IN ('$pmpro_page')
				AND post_type IN ('page','attachment')
				";

					$pages[$key] = $wpdb->get_results($sql, OBJECT);
				}
			}

			foreach ($pages as $key => $pmpro_pages) {
				$key_exists = pmpro_getOption($key . '_page_id');

				if (empty($key_exists) && !empty($pmpro_pages[0]->ID)) {
					pmpro_setOption($key . '_page_id', $pmpro_pages[0]->ID);
				}
			}
		}
	}
}
