<?php

/**
 * Streamit\Utility\Comments\Component class
 *
 * @package streamit
 */

namespace Streamit\Utility\Common;

use function Streamit\Utility\streamit;
use Streamit\Utility\Component_Interface;
use Streamit\Utility\Templating_Component_Interface;

/**
 * Class for managing comments UI.
 *
 * Exposes template tags:
 * * `streamit()->the_comments( array $args = array() )`
 *
 * @link https://wordpress.org/plugins/amp/
 */

class Component implements Component_Interface, Templating_Component_Interface
{
	/**
	 * Gets the unique identifier for the theme component.
	 *
	 * @return string Component slug.
	 */

	public $is_rtl;
	public $streamit_options;
	public $reset_massage = null; // For PMS Rest massage

	public function get_slug(): string
	{
		return 'common';
	}

	/**
	 * Adds the action and filter hooks to integrate with WordPress.
	 */

	public function initialize()
	{
		add_action("init", array($this, "streamit_add_global_prefix"));

		if (is_admin())
			add_action("export_wp", array($this, "streamit_modify_CPT_export_args"));

		add_filter('widget_tag_cloud_args', array($this, 'streamit_widget_tag_cloud_args'), 100);
		add_filter('wp_list_categories', array($this, 'streamit_categories_postcount_filter'), 100);
		add_filter('get_archives_link', array($this, 'streamit_style_the_archive_count'), 100);
		add_filter('get_the_archive_title_prefix', '__return_false');
		add_filter('upload_mimes', array($this, 'streamit_mime_types'), 100);
		add_action('wp_enqueue_scripts', array($this, 'streamit_remove_wp_block_library_css'), 100);
		add_action('wp_enqueue_scripts', array($this, 'streamit_remove_wp_block_library_css'), 100);
		add_action('wp_enqueue_scripts', array($this, 'streamit_remove_wp_block_library_css'), 100);
		add_filter('pre_get_posts', array($this, 'streamit_searchfilter'), 100);
		add_theme_support('post-formats', array(
			'aside',
			'image',
			'video',
			'quote',
			'link',
			'gallery',
			'audio',
		));
		add_action('admin_notices',  array($this, 'streamit_latest_version_announcement'));
		add_action('admin_enqueue_scripts', array($this, 'wpdocs_selectively_enqueue_admin_script'));
		add_action('wp_ajax_streamit_dismiss_notice', array($this, 'streamit_dismiss_notice'), 10);

		add_action('pms_recover_password_form_before_fields', array($this, 'streamit_recover_password_form_before_fields'));

		add_filter('login_form_top', function () {
			global $streamit_options;
			$logo = '';
			$options = isset($streamit_options['streamit_logo']['url']) ? $streamit_options['streamit_logo']['url'] : '';
			if (isset($options) && !empty($options)) {
				$logo = $this->streamit_get_svg($options, "img-fluid logo");
			}
			return '<a class="logo-link" href="' . esc_url(home_url('/')) . '">' . $logo . '</a>';
		}, 10);
		add_action('pms_register_form_top', function () {
			global $streamit_options;
			$logo = '';
			$options = isset($streamit_options['streamit_logo']['url']) ? $streamit_options['streamit_logo']['url'] : '';
			if (isset($options) && !empty($options)) {
				$logo = $this->streamit_get_svg($options, "img-fluid logo");
			}
			echo '<a class="logo-link" href="' . esc_url(home_url('/')) . '">' . $logo . '</a>';
		}, 10);

		add_filter('pms_recover_password_message', array($this, 'streamit_recover_password_message'), 10);
		add_filter('body_class', array($this, 'my_body_classes'));

		add_filter('wp_ulike_add_templates_args', array($this, 'streamit_default_counter_hide'));
		add_action('wp_ulike_after_up_vote_button', array($this, 'streamit_like_button_tooltip'), -50);
		add_filter('wp_ulike_ajax_counter_value', array($this, 'streamit_counter_structure'));
		add_action('redux/options/streamit_options/saved', array($this, 'wmpl_save_config_file'));
		add_action('wp', function () {
			is_singular(array('tv_show', 'episode')) && remove_action('template_redirect', 'redirect_canonical');
		});
		add_action('wp_head', array($this, 'streamit_add_share_meta'));
		add_filter('user_has_cap', array($this, 'streamit_add_user_access_to_edit_playlist'));
		// Add Filter to remove Season URL From Multiple Pagiation 
		add_filter('get_pagenum_link', function ($link) {
			return remove_query_arg('season', $link);
		});
		$this->is_rtl();
		add_action('streamit_layout_mode_add_attr', [$this, 'streamit_layout_mode_add_attr']);

		add_filter('streamit_login_form_title', function () {
			return '';
		});

		add_filter('login_form_middle', [$this, 'streamit_forgot_password'], 11);
		add_filter('login_form_bottom', [$this, 'streamit_social_login'], 11);

		if (class_exists('MasVideos')) {
			add_action('masvideos_before_single_movie_summary', array($this, 'streamit_go_back_button'), 9);
			add_action('masvideos_before_single_video_summary', array($this, 'streamit_go_back_button'), 9);
			add_action('masvideos_before_single_tvshow_summary', array($this, 'streamit_go_back_button'), 9);
			add_action('masvideos_before_single_episode_summary', array($this, 'streamit_go_back_button'), 9);
			add_action('masvideos_before_single_person_summary', array($this, 'streamit_go_back_button'), 9);
		}

		add_filter('post_class', array($this, 'streamit_add_go_back_class'));

		if (is_admin()) {
			add_action('admin_notices', array($this, 'iqonic_sale_banner_notice'));
			add_action('wp_ajax_iq_dismiss_notice', array($this, 'iq_dismiss_notice'));
		}
	}

	function streamit_add_global_prefix()
	{
		global $theme_prefix_php, $streamit_options;
		$theme_prefix_php = "streamit";

		if ($streamit_options === NULL) {
			$streamit_options = get_option("streamit_options");
		}
	}

	function streamit_modify_CPT_export_args()
	{
		$server_name = $_SERVER['SERVER_NAME'];

		if ($server_name == "wordpress.iqonic.design" || $server_name == "localhost" || $server_name == "192.168.1.179") {
			$remove_CPT_for_export = array(
				'custom_css',
				'wp_block',
				'wp_template',
				'wp_template_part',
				'wp_global_styles',
				'wp_navigation',
				'e-landing-page',
				'elementor_library',
				'acf-taxonomy',
				'acf-post-type',
				'acf-field-group',
				'acf-field',
				'tv_show_playlist',
				'video_playlist',
				'movie_playlist',
				'shop_order',
				'shop_order_refund',
				'shop_coupon'
			);

			apply_filters("streamit_modify_export_CPT", $remove_CPT_for_export);

			foreach ($remove_CPT_for_export as $CPT) {

				// Check if the post type is already registered.
				if (post_type_exists($CPT)) {
					// Get the existing arguments for the post type.
					$args = get_post_type_object($CPT);

					// Modify the can_export argument to false.
					$args->can_export = false;

					// Re-register the post type with the modified arguments.
					register_post_type($CPT, $args);
				}
			}
		}
	}

	public function streamit_add_go_back_class($classes)
	{
		array_push($classes, 'streamit-go-back');
		return $classes;
	}

	public function current_shortcode()
	{
		global $post;

		preg_match_all('/' . get_shortcode_regex() . '/', $post->post_content, $matches, PREG_SET_ORDER);
		$current_shortcode = '';
		foreach ($matches as $shortcode) {
			if (strpos($shortcode[0], '[' . $shortcode[2]) === 0) {
				$current_shortcode = $shortcode[2];
				break;
			}
		}

		return $current_shortcode;
	}

	public function streamit_forgot_password($middle)
	{
		$lost_password_url = wp_lostpassword_url();
		$current_shortcode = $this->current_shortcode();

		if ($current_shortcode == "pmpro_login") {
			$lost_password_url = esc_url(add_query_arg('action', urlencode('reset_pass'), pmpro_login_url()));
		}

		return $middle.'
			<div class="custom-links mt-0">
				<a href="' . esc_url($lost_password_url) . ' " class="iq-sub-card setting-dropdown">
				<h6 class="mb-0 ">' . esc_html__('Forgot Password?', "streamit") . '</h6>
				</a>
			</div>';
	}

	public function streamit_social_login($html)
	{
		global $streamit_options;

		$return_html = $iqonic_signup_link = $iqonic_signup_title = '';
		$current_shortcode = $this->current_shortcode();

		if ($current_shortcode == "iqonic_login" || $current_shortcode == "pmpro_login") {
			if (isset($streamit_options['streamit_signup_link'])) {
				$iqonic_signup_link = get_page_link($streamit_options['streamit_signup_link']);
				$iqonic_signup_title = $streamit_options['streamit_signup_title'];
			} else {
				if (function_exists('pmpro_getOption') && !empty(pmpro_getOption('levels_page_id'))) {
					$iqonic_signup_link = get_page_link(pmpro_getOption('levels_page_id'));
				}
			}

			if (empty($iqonic_signup_link)) return;

			$domain_name = get_bloginfo('name');

			$return_html .= '
				<div class="login-form-bottom">
				<div class="d-flex justify-content-center align-items-center links">'
				. esc_html__("New to ", "streamit") . $domain_name . '?
				<a href="' . esc_url($iqonic_signup_link) . '" class="iq-sub-card setting-dropdown  ml-2">
				<h6 class="mb-0 "> ';

			if (!empty($iqonic_signup_title)) {
				$return_html .= esc_html($iqonic_signup_title);
			} else {
				$return_html .= esc_html__('Register', 'streamit');
			}
			$return_html .= '</h6>
				</a>
				</div>';
		}

		if (shortcode_exists('miniorange_social_login') && !is_user_logged_in()) {
			$return_html .= '
			<div class="streamit-seperator">
				<span class="line-1"></span>
				<span class="or-section">' . esc_html__("OR", 'streamit') . ' </span>
				<span class="line-2"></span>
			</div>';

			$return_html .= '<div class="streamit-social-login-section">';
			$return_html .= do_shortcode('[miniorange_social_login]');
			$return_html .= '</div>';
		}
		$return_html .= '</div>';

		return $html . $return_html;
	}

	public function streamit_add_user_access_to_edit_playlist($all_caps)
	{
		global $streamit_options;
		if (isset($streamit_options['streamit_playlist_page']) && !empty($streamit_options['streamit_playlist_page']) && is_page($streamit_options['streamit_playlist_page'])) {
			$all_caps['edit_published_movie_playlists'] = true;
		}
		return $all_caps;
	}

	public function streamit_counter_structure($counter)
	{
		if (preg_replace('/[^0-9]/', '', $counter) < 10)
			return $counter;

		ob_start(); ?>
		<svg width="15" height="40" class="share-shape" viewBox="0 0 15 40" fill="none" xmlns="http://www.w3.org/2000/svg">
			<path fill-rule="evenodd" clip-rule="evenodd" d="M14.8842 40C6.82983 37.2868 1 29.3582 1 20C1 10.6418 6.82983 2.71323 14.8842 0H0V40H14.8842Z" fill="#191919"></path>
		</svg>
		<p class="m-0"><?php echo esc_html($counter); ?></p>
		<?php return  ob_get_clean();
	}

	public function wmpl_save_config_file($val)
	{
		if (class_exists('WooCommerce')) {
			$woof_setting = get_option('woof_settings');
			$woof_setting['per_page'] = (int)$val['woocommerce_product_per_page'];
			update_option('woof_settings', $woof_setting);
		}
	}
	public function my_body_classes($classes)
	{
		if (class_exists('ReduxFramework')) {
			global $streamit_options;
			$id = get_queried_object_id();

			$header_display = !empty($id) ? get_post_meta($id, 'name_header_display', true) : '';
			$page_header_layout = (is_plugin_active('advanced-custom-fields/acf.php') && !empty($id)) ? get_post_meta($id, 'header_layout_type', true) : '';

			if ($streamit_options['header_layout'] == 'custom' || $header_display === "yes" && $page_header_layout == 'custom' || (class_exists('WooCommerce') && (is_woocommerce() || is_cart() || is_checkout()) && isset($streamit_options['woo_header_layout']) && $streamit_options['woo_header_layout'] == 'custom')) {
				$classes = array_merge($classes, array('streamit-custom-header'));
			} else {
				$classes = array_merge($classes, array('streamit-default-header'));
			}
		} else {
			$classes = array_merge($classes, array('streamit-default-header'));
		}

		if (function_exists('pms_is_post_restricted') && pms_is_post_restricted(get_the_ID())) {
			$classes[] = 'is-restricted';
		}

		return $classes;
	}
	public function __construct()
	{
		add_filter('the_content', array($this, 'streamit_remove_empty_p'));
		add_filter('get_the_content', array($this, 'streamit_remove_empty_p'));
		add_filter('get_the_excerpt', array($this, 'streamit_remove_empty_p'));
		add_filter('the_excerpt', array($this, 'streamit_remove_empty_p'));
	}

	/**
	 * Gets template tags to expose as methods on the Template_Tags class instance, accessible through `streamit()`.
	 *
	 * @return array Associative array of $method_name => $callback_info pairs. Each $callback_info must either be
	 *               a callable or an array with key 'callable'. This approach is used to reserve the possibility of
	 *               adding support for further arguments in the future.
	 */
	public function template_tags(): array
	{
		return array(
			'streamit_pagination' 			=> array($this, 'streamit_pagination'),
			'streamit_get_embed_video' 		=> array($this, 'streamit_get_embed_video'),
			'streamit_get_svg' 				=> array($this, 'streamit_get_svg'),
			'streamit_more_content_js' 		=> array($this, 'streamit_more_content_js'),
			'streamit_content_description' 	=> array($this, 'streamit_content_description'),
			'streamit_is_rtl' 				=> array($this, 'streamit_is_rtl'),
			'streamit_get_restricted_content' => array($this, 'streamit_get_restricted_content'),
			'streamit_go_back_button' 		=> array($this, 'streamit_go_back_button'),
		);
	}

	public function streamit_more_content_js()
	{
		wp_enqueue_script("more-content", get_template_directory_uri() . '/assets/js/more-content.min.js',  array('jquery'), streamit()->get_version(), true);
	}

	public function streamit_go_back_button()
	{
		$previous_page_link = wp_get_referer();

		if (isset($previous_page_link) && !empty($previous_page_link)) { ?>
			<a href="<?php echo esc_url($previous_page_link); ?>" class="streamit-back-btn">
				<i class="fas fa-angle-left"></i>
			</a>
		<?php
		}
	}

	//return svg content or else image tag
	function streamit_get_svg($file, $class = "", $alt = "streamit")
	{
		global $wp_filesystem;
		$content =   '';

		require_once(ABSPATH . '/wp-admin/includes/file.php');
		WP_Filesystem();
		if ("svg" == wp_check_filetype($file)["ext"]) {
			$content = $wp_filesystem->get_contents($file);
		} else {
			$content = '<img class="' . esc_attr($class) . '" src="' . esc_url($file) . '" alt="' . esc_attr($alt) . '">';
		}
		return $content;
	}

	function streamit_get_embed_video($post_id)
	{
		$post = get_post($post_id);
		$content = do_shortcode(apply_filters('the_content', $post->post_content));
		$embeds = get_media_embedded_in_content($content);
		if (!empty($embeds)) {
			foreach ($embeds as $embed) {
				if (strpos($embed, 'video') || strpos($embed, 'youtube') || strpos($embed, 'vimeo') || strpos($embed, 'dailymotion') || strpos($embed, 'vine') || strpos($embed, 'wordPress.tv') || strpos($embed, 'embed') || strpos($embed, 'audio') || strpos($embed, 'iframe') || strpos($embed, 'object')) {
					return $embed;
				}
			}
		} else {
			return;
		}
	}

	function streamit_remove_empty_p($string)
	{
		return preg_replace('/<p>(?:\s|&nbsp;)*?<\/p>/i', '', $string);
	}

	function streamit_remove_wp_block_library_css()
	{
		wp_dequeue_style('wp-block-library-theme');
	}

	public function streamit_widget_tag_cloud_args($args)
	{
		$args['largest'] = 1;
		$args['smallest'] = 1;
		$args['unit'] = 'em';
		$args['format'] = 'list';

		return $args;
	}

	function streamit_mime_types($mimes)
	{
		$mimes['webp'] 		= 'image/webp';
		$mimes['svg'] 		= 'image/svg+xml';
		return $mimes;
	}

	function streamit_categories_postcount_filter($variable)
	{
		$variable = str_replace('<a', '<div class="wrap-cat"><a', $variable);
		$variable = str_replace('(', '<span class="post_count"> (', $variable);
		$variable = str_replace(')', ') </span></div>', $variable);
		return $variable;
	}

	function streamit_style_the_archive_count($links)
	{
		$links = str_replace('</a>&nbsp;(', '</a> <span class="archiveCount"> (', $links);
		$links = str_replace('&nbsp;)</li>', ' </li></span>', $links);
		return $links;
	}

	public function streamit_pagination($numpages = '', $pagerange = '', $paged = '', $wp_query = '', $add_args = false, $format = '?paged=%#%')
	{
		global $paged;

		if (empty($pagerange))
			$pagerange = 2;

		if (empty($paged)){
			$current_page = 1;
		}else{
			$current_page = $paged;
		}
			
		if ($numpages == '' && $wp_query == '') {
			global $wp_query;
			$numpages = $wp_query->max_num_pages;
			if (!$numpages) {
				$numpages = 1;
			}
		}

		if (!empty($wp_query)) {
			$paged = $wp_query->query_vars['paged'];
			$numpages = $wp_query->max_num_pages;
			$format = $format = '?paged=%#%';
		}

		/**
		 * We construct the pagination arguments to enter into our paginate_links
		 * function.
		 */
		$pagination_args = array(
			'format' 		=> $format,
			'total' 		=> $numpages,
			'current' 		=> $current_page,
			'show_all' 		=> false,
			'end_size' 		=> 1,
			'mid_size' 		=> $pagerange,
			'prev_next' 	=> true,
			'prev_text'     => '<i class="fas fa-chevron-left"></i>',
			'next_text'     => '<i class="fas fa-chevron-right"></i>',
			'type' 			=> 'list',
			'add_args' 		=> $add_args,
			'add_fragment' 	=> ''
		);

		$paginate_links = paginate_links($pagination_args);
		if ($paginate_links) {
			echo '<div class="col-lg-12 col-md-12 col-sm-12">
					<div class="pagination justify-content-center">
								<nav aria-label="Page navigation">';
			printf(esc_html__('%s', 'streamit'), $paginate_links);
			echo '</nav>
					</div>
				</div>';
		}
	}

	function streamit_searchfilter($query)
	{
		if (!is_admin()) {
			if ($query->is_search) {
				if (isset($_GET['post_type']) && $_GET['post_type']) {
					$query->set('post_type', $_GET['post_type']);
				} else if (isset($_GET['ajax_search']) && $_GET['ajax_search'] == 'true') {
					$query->set('post_type', apply_filters('streamit_search_filter_by_posttype', array('movie', 'tv_show', 'episode', 'video')));
				} else {
					$query->set('post_type', array('post'));
				}
			}

			if (is_archive() && $query->is_main_query() && isset($query->query_vars["post_type"]) &&  in_array($query->query_vars["post_type"], ["movie", "tv_show", 'video', 'episode'])) {
				$post_type = $query->query_vars["post_type"];
				$posts_per_page = $query->query_vars["posts_per_page"];
				unset($query->query_vars);
				$query->query_vars = ["post_type" => $post_type, "post_status" => "publish", "posts_per_page" => $posts_per_page];
			}
		}
		return $query;
	}

	public function streamit_latest_version_announcement()
	{
		global $current_user;
		$user_id = $current_user->ID;

		$template_version = get_option('template_version');
		if ($template_version && $template_version  <  wp_get_theme()->get('Version') && !get_user_meta($user_id, 'streamit_notification_5_0_0')) { ?>
			<div class="notice notice-warning streamit-notice  is-dismissible" id="streamit_notification_5_0_0">
				<div class="streamit-notice-main-box d-flex">
					<div class="streamit-notice-logo-push">
						<span><img src="<?php echo esc_url(get_template_directory_uri());  ?>/assets/images/redux/options.png"> </span>
					</div>
					<div class="streamit-notice-message">
						<h3><?php esc_html_e('Streamit v2.0.0 Is Here!', 'streamit'); ?></h3>
						<div class="streamit-notice-message-inner">
							<strong class="text-bold "><?php esc_html_e('Streamit Extensions Plugin Update Available.', 'streamit'); ?></strong>
							<a class="" href="<?php echo esc_url('https://assets.iqonic.design/documentation/wordpress/streamit-doc/index.html', 'streamit') ?>" target="_blank" rel="noopener noreferrer"><?php echo esc_html__('Click Here - How To Install Plugin', 'streamit') ?></a>
						</div>

					</div>
				</div>
				<div class="streamit-notice-cta">
					<button class="streamit-notice-dismiss streamit-dismiss-welcome notice-dismiss" data-msg="streamit_notification_5_0_0"><span class="screen-reader-text"><?php esc_html_e('Dismiss', 'streamit'); ?></span></button>
				</div>
			</div>
		<?php
		}
	}

	public function wpdocs_selectively_enqueue_admin_script()
	{
		wp_enqueue_script('admin-custom', get_template_directory_uri() . '/assets/js/admin-custom.min.js', array('jquery'), streamit()->get_version());
		wp_enqueue_style('admin-custom', get_template_directory_uri() . '/assets/css/admin-custom.min.css', array(), streamit()->get_version());
	}

	public 	function streamit_dismiss_notice()
	{
		global $current_user;
		$user_id = $current_user->ID;
		if (!empty($_POST['action']) && $_POST['action'] == 'streamit_dismiss_notice') {
			$theme_version = wp_get_theme()->get('Version');
			if (!empty(get_option('template_version'))) {

				add_option('template_version', $theme_version);
			} else {
				update_option('template_version', $theme_version);
			}

			add_user_meta($user_id, 'streamit_notification_5_0_0', 'true', true);
			wp_send_json_success();
		}
	}

	public function streamit_add_logo_in_pms_login_form()
	{
		global $streamit_options;
		$logo = '';
		if (is_plugin_active('advanced-custom-fields/acf.php') || class_exists('ReduxFramework')) {
			$acf_logo = get_post_meta(get_the_ID(), 'header_logo', true);

			if (!empty($acf_logo)) {
				$options = wp_get_attachment_url($acf_logo);
			} else if (isset($streamit_options['header_radio'])) {
				if ($streamit_options['header_radio'] == 1) {
					$logo_text = $streamit_options['header_text'];
					echo esc_html($logo_text);
				}
				if ($streamit_options['header_radio'] == 2) {
					$options = $streamit_options['streamit_mobile_logo']['url'];
				}
			}

			if (isset($options) && !empty($options)) {
				$logo = $this->streamit_get_svg($options, "img-fluid logo");
			}
		} elseif (has_header_image()) {
			$image = wp_get_attachment_image_src(get_theme_mod('custom_logo'), 'full');
			if (has_custom_logo()) {
				$logo = $this->streamit_get_svg($image, "img-fluid logo");
			} else {
				bloginfo('name');
			}
		} else {
			$logo_url = get_template_directory_uri() . '/assets/images/logo.png';
			$logo = $this->streamit_get_svg($logo_url, "img-fluid logo");
		}
		return '<a class="logo-link" href="' . esc_url(home_url('/')) . '">' . $logo . '</a>';
	}
	public function streamit_recover_password_message($pms_recover_notification)
	{
		$this->reset_massage = $pms_recover_notification;
	}

	public function streamit_recover_password_form_before_fields()
	{
		echo wp_kses($this->streamit_add_logo_in_pms_login_form(), 'post');
		echo wp_kses($this->reset_massage, 'post');
	}

	public static function streamit_get_restricted_content($id)
	{
		$search = array(esc_html__('view', 'streamit'), esc_html__('content', 'streamit'));
		$replace = array(esc_html__('download', 'streamit'), get_post_type($id));

		if (is_user_logged_in())
			echo str_replace($search, $replace, (esc_html__('The download of this file is restricted.', 'streamit')));
		else
			echo str_replace($search, $replace, (esc_html__('You must be logged in to view this content.', 'streamit')));
	}

	public function streamit_like_button_tooltip($wp_ulike_template)
	{
		if ($wp_ulike_template['total_likes'] < 10)
			return; ?>

		<span class="streamit-count-box">
			<?php echo esc_html__('9+', 'streamit'); ?>
		</span>

		<div class="streamit_tooltip count-box">
			<svg width="15" height="40" class="share-shape" viewBox="0 0 15 40" fill="none" xmlns="http://www.w3.org/2000/svg">
				<path fill-rule="evenodd" clip-rule="evenodd" d="M14.8842 40C6.82983 37.2868 1 29.3582 1 20C1 10.6418 6.82983 2.71323 14.8842 0H0V40H14.8842Z" fill="#191919" />
			</svg>
			<p class="m-0">
				<?php echo esc_html($wp_ulike_template['formatted_total_likes']); ?>
			</p>
		</div>
	<?php
	}

	public function streamit_default_counter_hide($args)
	{
		if ($args['total_likes'] >= 2) {
			$args['button_class'] .= ' streamit_default_counter_hide';
			$args['display_counters'] = true;
		}
		return $args;
	}

	public function streamit_add_share_meta()
	{ ?>
		<!-- Sharable Meta -->
		<meta property="og:type" content="article" />
		<meta property="og:title" content="<?php esc_attr(the_title()) ?>" />
		<meta property="og:url" content="<?php the_permalink() ?>" />
		<meta property="og:description" content="<?php echo esc_attr(get_the_excerpt()) ?>" />
		<meta property="article:published_time" content="<?php echo get_the_date('Y-m-d H:i:s'); ?>" />
		<meta property="article:modified_time" content="<?php echo get_the_modified_date('Y-m-d H:i:s'); ?>" />
		<meta property="og:site_name" content="<?php echo  get_bloginfo('name'); ?>" />
		<meta property="og:image" content="<?php echo empty(get_the_post_thumbnail_url()) ? get_template_directory_uri() . '/screenshot.png' : get_the_post_thumbnail_url() ?>" />
		<meta property="og:locale" content="<?php echo get_locale() ?>" />
		<meta name="twitter:text:title" content="<?php esc_attr(the_title()) ?>" />
		<meta name="twitter:card" content="summary_large_image" />
		<?php
	}

	public function streamit_content_description()
	{
		if (class_exists('Paid_Member_Subscription'))
			remove_filter('the_content', 'pms_filter_content', 11);

		remove_filter('the_content', 'pmpro_membership_content_filter', 5);
		remove_filter('the_content', 'pmpro_membership_content_filter', 15);

		the_content();

		add_filter('the_content', 'pmpro_membership_content_filter', 5);
		add_filter('the_content', 'pmpro_membership_content_filter', 15);

		if (class_exists('Paid_Member_Subscription'))
			add_filter('the_content', 'pms_filter_content', 11);
	}

	public function streamit_layout_mode_add_attr()
	{
		$layout_attr = '';
		if ($this->is_rtl) {
			$layout_attr = 'dir=';
			$layout_attr .= $this->is_rtl ? 'rtl' : 'ltr';
		}
		echo esc_attr($layout_attr);
	}

	public function is_rtl()
	{
		global $streamit_options;
		$dir = 'ltr';
		if (isset($_COOKIE['theme_scheme_direction']) && !empty($_COOKIE['theme_scheme_direction'])) {
			$dir = $_COOKIE["theme_scheme_direction"];
		} else if (class_exists('ReduxFramework') && !empty($streamit_options['streamit_layout_mode_options'])) {
			$dir = ($streamit_options['streamit_layout_mode_options'] == 2) ? 'rtl' : 'ltr';
		}
		$this->is_rtl = $dir == 'rtl';
	}
	public function streamit_is_rtl()
	{
		return $this->is_rtl;
	}

	public function iqonic_sale_banner_notice()
	{
		$type = "plugins";
		$product = "common";

		$get_sale_detail = get_transient('iq-notice');
		if (is_null($get_sale_detail) || $get_sale_detail === false) {
			$get_sale_detail = wp_remote_get("https://assets.iqonic.design/wp-product-notices/notices.json?ver=" . wp_rand());
			set_transient('iq-notice', $get_sale_detail, 3600);
		}

		if (!is_wp_error($get_sale_detail) && $content = json_decode(wp_remote_retrieve_body($get_sale_detail), true)) {
			if (get_option($content['data']['notice-id'], 0)) {
				return;
			}

			$currentTime =  current_datetime();
			if (($content['data']['start-sale-timestamp']  < $currentTime->getTimestamp() && $currentTime->getTimestamp() < $content['data']['end-sale-timestamp']) && isset($content[$type][$product])) { ?>
				<div class="iq-notice iq-sale-notice notice notice-success is-dismissible" style="padding: 0;">
					<a target="_blank" href="<?php echo esc_url($content[$type][$product]['sale-ink'] ?? "#")  ?>">
						<img src="<?php echo esc_url($content[$type][$product]['banner-img'] ?? "#")  ?>" style="object-fit: contain;padding: 0;margin: 0;display: block;" width="100%" alt="">
					</a>
					<input type="hidden" id="iq-notice-id" value="<?php echo esc_html($content['data']['notice-id']) ?>">
					<input type="hidden" id="iq-notice-nounce" value="<?php echo wp_create_nonce('iq-dismiss-notice') ?>">
				</div>
<?php
				wp_enqueue_script('iq-admin-notice', get_template_directory_uri() . '/assets/js/sale-notice.min.js', ['jquery'], false, true);
			}
		}
	}

	public function iq_dismiss_notice()
	{
		if (wp_verify_nonce($_GET['nounce'], 'iq-dismiss-notice')) {
			update_option($_GET['key'], 1);
		}
		wp_die();
	}
}
