<?php

/**
 * Streamit\Utility\Comments\Component class
 *
 * @package streamit
 */

namespace Streamit\Utility\Comments;

use function Streamit\Utility\streamit;
use Streamit\Utility\Component_Interface;
use Streamit\Utility\Templating_Component_Interface;

/**
 * Class for managing comments UI.
 *
 * Exposes template tags:
 * * `streamit()->the_comments( array $args = array() )`
 *
 * @link https://wordpress.org/plugins/amp/
 */

class Component implements Component_Interface, Templating_Component_Interface
{
	/**
	 * Gets the unique identifier for the theme component.
	 *
	 * @return string Component slug.
	 */

	public function get_slug(): string
	{
		return 'comments';
	}

	/**
	 * Adds the action and filter hooks to integrate with WordPress.
	 */

	public function initialize()
	{
		$mas_video_post_types = ['movie', 'tv_show', 'episode', 'video'];
		add_action('wp_enqueue_scripts', array($this, 'action_enqueue_comment_reply_script'));

		foreach ($mas_video_post_types as $post_type) {
			add_action('masvideos_' . $post_type . '_review_before', array($this, 'streamit_get_avatar_img'));
		}
	}

	/**
	 * Gets template tags to expose as methods on the Template_Tags class instance, accessible through `streamit()`.
	 *
	 * @return array Associative array of $method_name => $callback_info pairs. Each $callback_info must either be
	 *               a callable or an array with key 'callable'. This approach is used to reserve the possibility of
	 *               adding support for further arguments in the future.
	 */

	public function template_tags(): array
	{
		return array(
			'the_comments' => array($this, 'the_comments'),
		);
	}

	/**
	 * Enqueues the WordPress core 'comment-reply' script as necessary.
	 */

	public function action_enqueue_comment_reply_script()
	{

		// If the AMP plugin is active, return early.
		if (streamit()->is_amp()) {
			return;
		}

		// Enqueue comment script on singular post/page views only.
		if (is_singular() && comments_open() && get_option('thread_comments')) {
			wp_enqueue_script('comment-reply');
		}
	}

	/**
	 * Filters the comment form default arguments.
	 *
	 * Change the heading level to h2 when there are no comments.
	 *
	 * @param array $args The default comment form arguments.
	 * @return array      Modified comment form arguments.
	 */

	public function filter_comment_form_defaults(array $args): array
	{
		if (!get_comments_number()) {
			$args['title_reply_before'] = '<h4 id="reply-title" class="comment-reply-title">';
			$args['title_reply_after'] = '</h4>';
		}

		return $args;
	}

	/**
	 * Displays the list of comments for the current post.
	 *
	 * Internally this method calls `wp_list_comments()`. However, in addition to that it will render the wrapping
	 * element for the list, so that must not be added manually. The method will also take care of generating the
	 * necessary markup if amp-live-list should be used for comments.
	 *
	 * @param array $args Optional. Array of arguments. See `wp_list_comments()` documentation for a list of supported
	 *                    arguments.
	 */

	public function the_comments(array $args = array())
	{
		$args = array_merge(
			$args,
			array(
				'style' => 'ol',
				'short_ping' => true,
			)
		);

		$amp_live_list = streamit()->using_amp_live_list_comments();

		if ($amp_live_list) {
			$comment_order = get_option('comment_order');
			$comments_per_page = get_option('page_comments') ? (int)get_option('comments_per_page') : 10000;
			$poll_inverval = MINUTE_IN_SECONDS * 1000; ?>

			<amp-live-list id="amp-live-comments-list-<?php the_ID(); ?>" <?php echo ('asc' === $comment_order) ? ' sort="ascending" ' : ''; ?> data-poll-interval="<?php echo esc_attr($poll_inverval); ?>" data-max-items-per-page="<?php echo esc_attr($comments_per_page); ?>">
			<?php } ?>
			<ol class="commentlist" <?php echo esc_html($amp_live_list) ? ' items' : ''; ?>>
				<?php wp_list_comments(array(
					'walker'      => new Component_Walker_Comment,
					'style'      => 'ol',
					'avatar_size' => 80,
				)); ?>
			</ol><!-- .comment-list -->

			<?php the_comments_navigation();

			if ($amp_live_list) { ?>
				<div>
					<button class="button" on="tap:amp-live-comments-list-<?php the_ID(); ?>.update"><?php esc_html_e('New comment(s)', 'streamit'); ?></button>
				</div>
			</amp-live-list>
			<?php
			}
		}

		/**
		 * Adds a pagination reference point attribute for amp-live-list when theme supports AMP.
		 *
		 * This is used by the navigation_markup_template filter in the comments template.
		 *
		 * @link https://www.ampproject.org/docs/reference/components/amp-live-list#pagination
		 *
		 * @param string $markup Navigation markup.
		 * @return string Filtered markup.
		 */

		public function filter_add_amp_live_list_pagination_attribute(string $markup): string
		{
			return preg_replace('/(\s*<[a-z0-9_-]+)/i', '$1 pagination ', $markup, 1);
		}

		public function streamit_get_avatar_img($comment)
		{
			if ($img_link = get_user_meta($comment->user_id, 'streamit_profile_image', true))
				printf('<img alt="" src="%s" class="avatar avatar-60 photo" loading="lazy" width="60" height="60">', $img_link);
			else
				echo get_avatar($comment, apply_filters('masvideos_episode_review_gravatar_size', '60'), '');
		}
	}

	if (!class_exists('Component_Walker_Comment')) {
		class Component_Walker_Comment extends \Walker_Comment
		{
			function end_el(&$output, $comment, $depth = 0, $args = array())
			{
				if (!empty($args['end-callback'])) {
					ob_start();
					call_user_func($args['end-callback'], $comment, $args, $depth);
					$output .= ob_get_clean();
					return;
				}
				if ('div' === $args['style']) {
					$output .= "</div><!-- #comment-## -->\n";
				}
			}

			public function html5_comment($comment, $depth, $args)
			{
				switch ($comment->comment_type):
					case 'pingback':
					case 'trackback':
						if (isset($args['style']) && 'div' == $args['style']) {
							$tag = 'div';
							$add_below = 'comment';
						} else {
							$tag = 'li';
							$add_below = 'div-comment';
						} ?>
					<li <?php comment_class('iq-comments-item'); ?> id="comment-<?php comment_ID(); ?>">
						<h5 class="mt-0 mb-0"><span class="iq-type-date"><span class="mr-2"><?php echo esc_html__(comment_type() . ':', 'streamit'); ?></span><?php comment_author_link(); ?></span> <?php edit_comment_link(esc_html__('(Edit)', 'streamit'), '<span class="edit-link">', '</span>'); ?>
						</h5>

					</li>
				<?php
						break;
					default:
						global $post; ?>

					<li <?php comment_class(); ?> id="li-comment-<?php comment_ID(); ?>">
						<article id="div-comment-<?php comment_ID(); ?>" class="comment-body">
							<div class="iq-comments-media">
								<div class="iq-comment-wrap">
									<div class="iq-comments-photo">
										<?php $profile_image = get_user_meta($comment->user_id, 'streamit_profile_image', true);
										if (!empty($profile_image)) {
											echo '<img src="' . esc_url($profile_image) . '" class="avatar avatar-80 photo img-fluid rounded-circle" />';
										} else {
											echo get_avatar($comment, $args['avatar_size']);
										} ?>
									</div>

									<div class="iq-comments-info">
										<!-- .comment-metadata -->
										<div class="wrap-commentitle">
											<div class="main-wrap-base d-flex align-items-center">
												<h5 class="title">
													<?php printf(wp_kses('%s ', 'streamit'), sprintf('%s', get_comment_author_link())); ?>
												</h5>

												<?php edit_comment_link(esc_html__('Edit', 'streamit'), '<span class="d-inline-block ml-2">', '</span>'); ?>
											</div>


											<div class="iq-comment-metadata">
												<a href="<?php echo esc_url(get_comment_link($comment->comment_ID)); ?>">
													<time datetime="<?php comment_time('c'); ?>">
														<?php printf(wp_kses('%1$s', '1: date'), get_comment_date()); ?>
													</time>
												</a>
											</div>
										</div>


										<?php if ('0' == $comment->comment_approved) : ?>
											<p class="comment-awaiting-moderation"><?php esc_html__('Your comment is awaiting moderation.', 'streamit'); ?></p>
										<?php endif; ?>

										<div class="comment-content">
											<?php comment_text(); ?>
										</div><!-- .comment-content -->

										<div class="reply iq-reply iq-button-style-2 has-icon btn-icon-right">
											<?php
											$args["reply_text"] = '<svg xmlns="http://www.w3.org/2000/svg" width="16" height="12" viewBox="0 0 16 12" fill="none"><path d="M13.0287 5.55C11.6004 4.12174 9.56563 3.33913 7.2178 3.20217V1.1087C7.2178 0.795652 6.98302 0.502174 6.68954 0.384783C6.39606 0.267391 6.04389 0.326087 5.8091 0.56087L0.996059 5.45217C0.702581 5.76522 0.683016 6.25435 0.996059 6.54783L5.8091 11.4391C6.02432 11.6739 6.39606 11.7326 6.68954 11.6152C6.98302 11.4978 7.2178 11.2043 7.2178 10.8913V8.77826C8.78302 8.73913 12.4613 8.89565 13.8113 11.2826C13.9482 11.537 14.2026 11.6739 14.4765 11.6739C14.5352 11.6739 14.5939 11.6739 14.6721 11.6543C15.0243 11.5565 15.2787 11.263 15.2787 10.8913C15.2591 10.7739 15.2591 7.78044 13.0287 5.55ZM6.3178 7.25217C5.9265 7.29131 5.65258 7.62391 5.65258 8.03478V8.97391L2.7178 6L5.65258 3.02609V3.96522C5.65258 4.39565 5.96562 4.74783 6.39606 4.74783C10.6026 4.74783 12.383 6.76304 13.1461 8.48478C10.3874 6.84131 6.51345 7.23261 6.3178 7.25217Z" fill="#E50914"/></svg>'
												.  esc_html__('Reply', 'streamit');
											$iq_class = 'btn-links';
											echo preg_replace(
												'/comment-reply-link/',
												'comment-reply-link ' . $iq_class,
												get_comment_reply_link(array_merge($args, array(
													'add_below' => 'div-comment',
													'depth' => $depth,
													'max_depth' => $args['max_depth'],
												))),
												1
											); ?>
										</div>
										<!-- .reply -->
									</div><!-- .comment-author -->
								</div>
							</div>
						</article><!-- .comment-body -->
					</li>
<?php
						break;
				endswitch;
			}
		}
		// end of WPSE_Walker_Comment
	}
?>