<?php

// Exit if accessed directly
if (!defined('ABSPATH')) exit;
if (!class_exists('PMS_Payment_Gateway') || !is_plugin_active('woocommerce/woocommerce.php')) return;

/**
 * Woocomerce Payment Gateway
 *
 * Provides an Woocomerce Payment Gateway; mainly for testing purposes.
 * We load it later to ensure WC is loaded first since we're extending it.
 *
 * @class 		Wp_Rig_WC_Payment_Gateway
 * @extends		WC_Payment_Gateway
 * @version		1.0.0
 * @package     PMS_Payment_Gateway
 * @author 		iqonic
 **/

class Wp_Rig_WC_Payment_Gateway extends PMS_Payment_Gateway
{
    /*
     * The payment gateway slug
     *
     * @access public
     * @var string
     *
     */
    public $payment_gateway = 'streamit_wc_payment_gateway';

    private $subscription_product_id;

    /**
     * The features supported by the payment gateway
     *
     * @access public
     * @var array
     *
     */
    public $supports;

    public $user_order_url;


    public function init()
    {

        // Filter for Profile Builder Plugin Url Redirect
        add_filter('wppb_register_redirect', function ($url) {
            if ($this->user_order_url) {
                $url = $this->user_order_url . '&' . explode('?', $url)[1];
            }
            return $url;
        });

        $this->supports = apply_filters('pms_gateway_manual_supports', array(
            'subscription_sign_up_fee',
            'subscription_free_trial',
        ));
        add_action('woocommerce_order_status_completed', array($this, 'streamit_subcription_purchased_payment_status'), 99, 1);
        add_action('woocommerce_order_status_pending', array($this, 'streamit_subcription_purchased_payment_status'), 99, 1);
        add_action('woocommerce_order_status_failed', array($this, 'streamit_subcription_purchased_payment_status'), 99, 1);
        add_action('woocommerce_order_status_refunded', array($this, 'streamit_subcription_purchased_payment_status'), 99, 1);

        add_filter('woocommerce_thankyou_order_received_text', array($this, 'streamit_subcription_order_massage'), 10, 2);
    }
    
    public function streamit_subcription_order_massage($msg, $order)
    {
        foreach ($order->get_items() as $item_id => $item) {
            $product = $item->get_product();
            $product_id = $product->get_id();
            if (get_post_meta($product_id, '_is_subscription', true)) {
                $order_pms = get_post_meta($order->get_id(), '_pms', true);
                $payment_id = $order_pms['pms_payment_id'];
                $payment = pms_get_payment($payment_id);

                return $this->success_messages($payment->status, get_post_meta($order->get_id(), 'pms_action', true));
            }
        }
        return esc_html__($msg, 'streamit-extensions');
    }

    public function streamit_subcription_purchased_payment_status($order_id)
    {
        if (!$order_id) {
            return;
        }
        $order = wc_get_order($order_id);

        // Loop through order items
        foreach ($order->get_items() as $item_id => $item) {
            // Get the product object
            $product = $item->get_product();

            // Get the product Id
            $product_id = $product->get_id();

            if (get_post_meta($product_id, '_is_subscription', true)) {
                $order_pms = get_post_meta($order->get_id(), '_pms', true);

                $payment_id = $order_pms["pms_payment_id"];
                $payment = pms_get_payment($payment_id);
                $payment->update(array('status' => $order->get_status(), 'transaction_id' => $order->get_transaction_id()));

                $this->activate_member_subscription($payment, $order->get_status());
            }
        }
    }


    /*
     * Process payment
     *
     */
    public function process_sign_up()
    {
        $subscription = pms_get_current_subscription_from_tier($this->user_id, $this->subscription_data['subscription_plan_id']);

        if (empty($subscription) && isset($_POST['pms_current_subscription']))
            $subscription = pms_get_member_subscription(absint($_POST['pms_current_subscription']));

        // Activate subscription if plan has a free trial
        if (!empty($this->subscription_data['trial_end'])) {

            $subscription->update(array('status' => 'active', 'billing_next_payment' => $this->subscription_data['trial_end'], 'billing_amount' => $this->subscription_plan->price));

            pms_add_member_subscription_log($subscription->id, 'subscription_trial_started', array('until' => $this->subscription_data['trial_end']));

            pms_add_member_subscription_log($subscription->id, 'subscription_activated');
        }

        if ($this->recurring) {

            $billing_next_payment = !empty($this->subscription_data['trial_end']) ?  $this->subscription_data['trial_end'] : $this->subscription_data['expiration_date'];

            $subscription_data = array(
                'billing_next_payment'  => $billing_next_payment,
                'billing_duration'      => $this->subscription_plan->is_fixed_period_membership() ? '1' : $this->subscription_plan->duration,
                'billing_duration_unit' => $this->subscription_plan->is_fixed_period_membership() ? 'year' : $this->subscription_plan->duration_unit,
                'billing_amount'        => $this->subscription_data['billing_amount'],
            );
            $subscription->update($subscription_data);
        }

        do_action('pms_manual_gateway_signup_processed', $this->subscription_data, $this->payment_id);
        $product_id = $this->subscription_product_id = $this->streamit_subcription_woocommerce_product();


        // Now we create the order
        $order = wc_create_order();

        $obj_product = wc_get_product($product_id);

        $order->add_product($obj_product);

        // This is an existing SIMPLE product
        $order->set_customer_id($this->user_id);
        $order->calculate_totals();
        $order->update_status("Completed", 'Imported order', TRUE);

        update_post_meta($order->get_id(), '_pms', array('pms_payment_id' => $this->payment_id, 'pms_action' => isset($_GET['pms-action']) ? $_GET['pms-action'] : false));

        $payment = pms_get_payment($this->payment_id);

        $payment->update(array('status' => 'pending', 'transaction_id' => $order->get_id()));

        if (isset($_POST['pms_register']) || (isset($_POST['pmstkn']) && isset($_POST['pay_gate']) && $_POST['pay_gate'] == "woocommerce_payment_gateway")) {
            $user =  array('user_login' => (isset($_POST['user_login']) ? sanitize_user($_POST['user_login']) : ''), 'user_password' => (isset($_POST['pass1']) ? $_POST['pass1'] : ''), 'remember' => true);
            wp_signon($user, is_ssl());
            wp_redirect($order->get_checkout_payment_url());
            exit;
        } else {
            $this->user_order_url = $order->get_checkout_payment_url();
        }
    }
    /**
     * post_exists_by_slug
     * @param $post_slug
     * @param string $type
     *
     * @return number false if no post exists; post ID otherwise.
     */
    public function post_exists_by_slug($post_slug, $type = 'post')
    {
        $args_posts = array(
            'post_type'      => $type,
            'name'           => $post_slug,
            'posts_per_page' => 1,
        );
        $loop_posts = new WP_Query($args_posts);
        if (!$loop_posts->have_posts()) {
            $return_val = false;
        } else {
            $loop_posts->the_post();
            $return_val = $loop_posts->post->ID;
        }
        wp_reset_postdata();
        return $return_val;
    }
    /**
     * streamit_subcription_woocommerce_product
     *
     * @return int|bool false if no post exists; post ID otherwise.
     */
    public function streamit_subcription_woocommerce_product()
    {
        $post = get_post($this->subscription_data['subscription_plan_id']);

        $sub_post_name = get_post_field('post_name', $post);
        $sub_product = $this->post_exists_by_slug($sub_post_name, 'product');
        if ($sub_product) {
            update_post_meta($sub_product, '_sale_price', $this->amount);
            update_post_meta($sub_product, '_price', $this->amount);
            update_post_meta($sub_product, '_visibility', 'hidden');
            $terms = array('exclude-from-search', 'exclude-from-catalog'); // for hidden..
            wp_set_post_terms($sub_product, $terms, 'product_visibility', false);
            return $sub_product;
        }

        $postarr = array(
            'post_title'    => get_the_title($post),
            'post_type'     => 'product',
            'post_status'   => 'publish',
            'post_name' => $sub_post_name
        );


        $streamit_product_id = wp_insert_post($postarr);

        update_post_meta($streamit_product_id, '_visibility', 'hidden');
        $terms = array('exclude-from-search', 'exclude-from-catalog'); // for hidden..
        wp_set_post_terms($streamit_product_id, $terms, 'product_visibility', false);
        update_post_meta($streamit_product_id, '_stock_status', 'instock');
        update_post_meta($streamit_product_id, 'total_sales', '0');
        update_post_meta($streamit_product_id, '_downloadable', 'yes');
        update_post_meta($streamit_product_id, '_virtual', 'yes');
        update_post_meta($streamit_product_id, '_regular_price', '');
        update_post_meta($streamit_product_id, '_sale_price', $this->amount);
        update_post_meta($streamit_product_id, '_purchase_note', '');
        update_post_meta($streamit_product_id, '_featured', 'no');
        update_post_meta($streamit_product_id, '_weight', '');
        update_post_meta($streamit_product_id, '_length', '');
        update_post_meta($streamit_product_id, '_width', '');
        update_post_meta($streamit_product_id, '_height', '');
        update_post_meta($streamit_product_id, '_sku', '');
        update_post_meta($streamit_product_id, '_product_attributes', []);
        update_post_meta($streamit_product_id, '_sale_price_dates_from', '');
        update_post_meta($streamit_product_id, '_sale_price_dates_to', '');
        update_post_meta($streamit_product_id, '_price', $this->amount);
        update_post_meta($streamit_product_id, '_sold_individually', 'yes');
        update_post_meta($streamit_product_id, '_manage_stock', 'no');
        update_post_meta($streamit_product_id, '_backorders', 'no');
        update_post_meta($streamit_product_id, '_is_subscription', true);
        update_post_meta($streamit_product_id, 'subscription_id', $this->subscription_product_id);
        wc_update_product_stock($streamit_product_id, 0, 'set');
        update_post_meta($streamit_product_id, '_stock', '');

        return $streamit_product_id;
    }
    // Adds a pending payment for the user when there's a free trial
    public function process_payment($payment_id = 0, $subscription_id = 0)
    {
        $subscription = pms_get_member_subscription($subscription_id);

        $data = array(
            'status'               => 'pending',
            'billing_next_payment' => '',
            'trial_end'            => ''
        );

        $subscription->update($data);

        pms_add_member_subscription_log($subscription->id, 'subscription_trial_end');

        return false;
    }


    /*
     * Change the default success message for the different payment actions
     *
     * @param string $message
     * @param string $payment_status
     * @param string $payment_action
     * @param obj $payment
     *
     * @return string
     *
     */
    public function success_messages($payment_status, $payment_action, $message = 'Thank you for Purchase')
    {
        // We're interested in changing only the success messages for paid subscriptions
        // which will all have the "pending" status
        switch ($payment_action) {

            case 'upgrade_subscription':
                $message = __('Thank you for upgrading. The changes will take effect after the payment is received.', 'paid-member-subscriptions');
                break;

            case 'downgrade_subscription':
                $message = __('Thank you for downgrading. The changes will take effect after the payment is received.', 'paid-member-subscriptions');
                break;

            case 'change_subscription':
                $message = __('Thank you for choosing another plan. The changes will take effect after the payment is received.', 'paid-member-subscriptions');
                break;

            case 'renew_subscription':
                $message = __('Thank you for renewing. The changes will take effect after the payment is received.', 'paid-member-subscriptions');
                break;

            case 'new_subscription':
                $message = __('Thank you for subscribing. The subscription will be activated after the payment is received.', 'paid-member-subscriptions');
                break;

            case 'retry_payment':
                $message = __('The subscription will be activated after the payment is received.', 'paid-member-subscriptions');
                break;

            default:
                break;
        }

        return $message;
    }


    /**
     * Activates the member's account when the payment is marked as complete.
     * If the subscription is already active, add the extra time to the subscription expiration date
     *
     * @param int   $payment_id
     * @param array $data         - an array with modifications made when saving the payment in the back-end
     * @param array $old_data     - the array of values representing the payment before the update
     *
     * @return void
     *
     */
    public function activate_member_subscription($payment, $status)
    {
        // Get plan
        $subscription_plan = pms_get_subscription_plan($payment->subscription_id);

        // The subscription plan ID from the payment matches an existing subscription for this user
        $member_subscriptions = pms_get_member_subscriptions(array('user_id' => $payment->user_id, 'subscription_plan_id' => $payment->subscription_id, 'number' => 1));

        if (!empty($member_subscriptions)) {

            $member_subscription = $member_subscriptions[0];

            $subscription_plan = pms_get_subscription_plan($payment->subscription_id);

            if ($member_subscription->status == 'active') {
                if ($subscription_plan->is_fixed_period_membership()) {
                    if ($subscription_plan->fixed_period_renewal_allowed())
                        $member_subscription->update(array('expiration_date' => date('Y-m-d H:i:s', strtotime(pms_sanitize_date($member_subscription->expiration_date) . '+ 1 year'))));
                    else
                        $member_subscription->update(array('expiration_date' => date('Y-m-d H:i:s', strtotime(pms_sanitize_date($member_subscription->expiration_date)))));
                } else {
                    $member_subscription->update(array('expiration_date' => date('Y-m-d H:i:s', strtotime(pms_sanitize_date($member_subscription->expiration_date) . '+' . $subscription_plan->duration . ' ' . $subscription_plan->duration_unit))));
                }
            } else if ($member_subscription->status == 'expired') {
                if ($subscription_plan->is_fixed_period_membership()) {
                    if ($subscription_plan->fixed_period_renewal_allowed())
                        $member_subscription->update(array('status' => 'active', 'expiration_date' => date('Y-m-d H:i:s', strtotime(pms_sanitize_date($member_subscription->expiration_date) . '+ 1 year'))));
                    else
                        $member_subscription->update(array('status' => 'active', 'expiration_date' => date('Y-m-d H:i:s', strtotime(pms_sanitize_date($member_subscription->expiration_date)))));
                } else {
                    $member_subscription->update(array('status' => 'active', 'expiration_date' => date('Y-m-d H:i:s', strtotime(date('Y-m-d H:i:s') . '+' . $subscription_plan->duration . ' ' . $subscription_plan->duration_unit))));
                }
            } else if ($member_subscription->status == 'canceled') {
                if (strtotime($member_subscription->expiration_date) > strtotime('now')) {
                    if ($subscription_plan->is_fixed_period_membership()) {
                        if ($subscription_plan->fixed_period_renewal_allowed())
                            $timestamp = strtotime(pms_sanitize_date($member_subscription->expiration_date) . '+ 1 year');
                        else
                            $timestamp = strtotime(pms_sanitize_date($member_subscription->expiration_date));
                    } else {
                        $timestamp = strtotime(pms_sanitize_date($member_subscription->expiration_date) . '+' . $subscription_plan->duration . ' ' . $subscription_plan->duration_unit);
                    }
                } else {
                    if ($subscription_plan->is_fixed_period_membership()) {
                        if ($subscription_plan->fixed_period_renewal_allowed())
                            $timestamp = strtotime(pms_sanitize_date($member_subscription->expiration_date) . '+ 1 year');
                        else
                            $timestamp = strtotime(pms_sanitize_date($member_subscription->expiration_date));
                    } else {
                        $timestamp = strtotime(date('Y-m-d H:i:s') . '+' . $subscription_plan->duration . ' ' . $subscription_plan->duration_unit);
                    }
                }
                $member_subscription->update(array('status' => 'active', 'expiration_date' => date('Y-m-d H:i:s', $timestamp)));
            } else
                $member_subscription->update(array('status' => 'active'));

            pms_add_member_subscription_log($member_subscription->id, 'admin_subscription_activated_payments');
            return;
        }
        // The plan from the payment is not the current user plan
        $old_subscription = pms_get_member_subscription($payment->member_subscription_id);

        if (!empty($old_subscription) && !empty($old_subscription->id)) {

            $old_plan_id = $old_subscription->subscription_plan_id;

            $subscription_plan = pms_get_subscription_plan($payment->subscription_id);

            $subscription_data = array(
                'user_id'              => $payment->user_id,
                'subscription_plan_id' => $subscription_plan->id,
                'start_date'           => date('Y-m-d H:i:s'),
                'expiration_date'      => $subscription_plan->get_expiration_date(),
                'status'               =>  $status == 'completed' ? 'active' : 'pending',
                'payment_gateway'      => $this->payment_gateway,
            );

            // reset custom schedule
            if ($old_subscription->payment_gateway != 'manual') {
                $subscription_data['billing_amount'] = '';
                $subscription_data['billing_duration'] = '';
                $subscription_data['billing_duration_unit'] = '';
                $subscription_data['billing_next_payment'] = '';
            }

            $old_subscription->update($subscription_data);

            $context = pms_get_change_subscription_plan_context($old_plan_id, $subscription_plan->id);

            pms_add_member_subscription_log($old_subscription->id, 'subscription_' . $context . '_success', array('old_plan' => $old_plan_id, 'new_plan' => $subscription_plan->id));
        }
    }

    public function remove_retry_payment($output, $subscription_plan, $subscription)
    {
        if (!empty($subscription['payment_gateway']) && $subscription['payment_gateway'] == 'manual')
            return;

        return $output;
    }

    /**
     * For manual gateway payments, do not let users request a Renewal more than once.
     *
     * @param  string    $output
     * @param  object    $subscription_plan
     * @param  array     $subscription
     * @param  int       $user_id
     * @return string|void
     */
    public function remove_renewal_action($output, $subscription_plan, $subscription, $user_id)
    {

        $payments = pms_get_payments(array('user_id' => $user_id, 'subscription_plan_id' => $subscription_plan->id));

        if (!empty($payments)) {
            foreach ($payments as $payment) {
                if ($payment->payment_gateway == 'manual' && $payment->status == 'pending' && $payment->type == 'subscription_renewal_payment')
                    return;
            }
        }

        return $output;
    }

    public function change_free_trial_payment_type($payment_data, $subscription)
    {

        if ($subscription->payment_gateway == 'manual')
            $payment_data['type'] = 'subscription_initial_payment';

        return $payment_data;
    }

    public function add_mark_as_completed_action($actions, $item)
    {
        if (empty($item['id']))
            return $actions;

        $payment = pms_get_payment($item['id']);

        if (empty($payment->id))
            return $actions;

        if ($payment->payment_gateway != 'manual' || $payment->status != 'pending')
            return $actions;

        $delete_action = $actions['delete'];
        unset($actions['delete']);

        $actions['complete_payment'] = '<a href="' . wp_nonce_url(add_query_arg(array('pms-action' => 'complete_payment', 'payment_id' => $item['id'])), 'pms_payment_nonce') . '">' . __('Complete Payment', 'paid-member-subscriptions') . '</a>';

        $actions['delete'] = $delete_action;

        return $actions;
    }
}

add_filter('pms_payment_gateways', function ($payment_gateways = array()) {
    if (empty($payment_gateways)) return $payment_gateways;
    global $streamit_options;

    $woo_payment_gateway_title = isset($streamit_options['streamit_woocommerce_payment_gateway_title']) && !empty($streamit_options['streamit_woocommerce_payment_gateway_title']) ? $streamit_options['streamit_woocommerce_payment_gateway_title'] : 'Woocommerce Payment Gateway';
    $payment_gateways['woocommerce_payment_gateway'] = array(
        'display_name_user' => $woo_payment_gateway_title,
        'display_name_admin' => $woo_payment_gateway_title,
        'class_name' => 'Wp_Rig_WC_Payment_Gateway'
    );
    return $payment_gateways;
});
